; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;   Unauthorized reproduction prohibited.
;+
; NAME:
;   PRO p_delete_comment, header, pos_or_tag
;
; PURPOSE:
;   Delete a comment in a PDS header
;
; CALLING SEQUENCE:
;   PRO p_delete_comment, header, pos_or_tag
;
; INPUTS:
;   header:     The PDS header structure to be modified
;   pos_or_tag: If integer value the input field means the absolute location (linenumber)
;               in the header where the comment should be found
;               If string value then field refers to a label tag
;               and the comment will be delete just before the location of the tag
;
; KEYWORD PARAMETERS:
;   AFTER:                Is set the comment will be inserted after the specified label
;                         (Only valid for string values of pos_or_tag)
;   RELATIVE_SHIFT:       Add to index for final position
;   MATCH:                If set pos_or_tag is interpreted as a string the must exists in the comment
;                         If set all comments containing the match string will be deleted  
;   CLEAR_AFTER:          If set all comments after the specified index (inclusive) will be cleared
;   CLEAR_BEFORE:         If set all comments before the specified index (inclusive) will be cleared
;   CLEAR_ALL:            If set all comments in the header will be deleted 
; OUTPUTS:
;   header:    Will contain the modified header structure
;
; RETURNS:
;   Nothing
;
; RESTRICTIONS:
;    The function can only be used for single line comments
;    Using it on multiline comments will probably cause curroptions
;
; EXAMPLE:
;   h =p_new_header()
;   p_setvalue, h, "TAG1", 0
;   p_setvalue, h, "TAG2", 1
;   p_set_comment, h, "TAG2", "Hello World"
;   p_delete_comment, h, "TAG2"
;
; MODIFICATION HISTORY:
;   Written by:  Stubbe F. Hviid, 12/05-2004
;-

PRO p_delete_comment, header, pos_or_tag, $
                                          MATCH=MATCH, $
                                          AFTER=AFTER, RELATIVE_SHIFT=RELATIVE_SHIFT, $
                                          CLEAR_AFTER=CLEAR_AFTER, CLEAR_BEFORE=CLEAR_BEFORE, $
                                          CLEAR_ALL=CLEAR_ALL
   
  ; first handle /MATCH
  if keyword_set(MATCH) then begin
    for i=n_elements(header.tags)-1, 0, -1 do begin
      if header.tags[i] eq '' then begin
        if strpos(header.values[i], pos_or_tag) ge 0 then begin
          p_delete_tag, header, fix(i)
        endif   
      endif
    endfor
    
    return  ; exit point
  endif
   
   
  ; first handle tag input 
  if size(pos_or_tag, /type) eq 7 then begin
    tmp = p_value(header, pos_or_tag, FOUND=FOUND, INDEX=INDEX)
    if FOUND eq 0 then begin
      print, 'ERROR: Cannot locate tag: ' + strtrim(pos_or_tag)
      return
    endif
    pos_or_tag = INDEX
    if keyword_set(AFTER) then pos_or_tag++ else pos_or_tag--
    if pos_or_tag lt 0 then pos_or_tag = 0
  endif
  
  ; set initial index
  index = pos_or_tag
  
  ; handle relative shift
  if n_elements(RELATIVE_SHIFT) then index += RELATIVE_SHIFT

  ; handle special cases
  if keyword_set(CLEAR_AFTER) then begin
    for i=n_elements(header.tags)-1, index, -1 do begin
      if header.tags[i] eq '' then p_delete_tag, header, fix(i)
    endfor
    return 
  endif
  
  if keyword_set(CLEAR_BEFORE) then begin
    for i=index, 0, -1 do begin
      if header.tags[i] eq '' then p_delete_tag, header, fix(i)
    endfor
    return 
  endif
  
  if keyword_set(CLEAR_ALL) then begin
    for i=n_elements(header.tags)-1, 0, -1 do begin
      if header.tags[i] eq '' then p_delete_tag, header, fix(i)
    endfor
    return 
  endif
  
  ; nominal single comment delete
  if header.tags[index] eq '' then p_delete_tag, header, index
END