; Copyright (c) 2005, Stubbe F. Hviid
;   All rights reserved
;+
; NAME:
;   FUNCTION p_read_text_file, file
;
; PURPOSE:
;   Read the content of text file into a string array
;
; CATEGORY:
;   file io
;
; CALLING SEQUENCE:
;   FUNCTION p_read_text_file, file
;
; INPUTS:
;   file: filename of file to be read (if not specified then a pickfile dialog will be shown)
;
; OPTIONAL INPUTS:
;   None
;
; KEYWORD PARAMETERS:
;   STRIP_PDS_HEADER:		If set then any PDS labels will be stripped from the text before returning
;   LABEL					If a PDS label existed in the file then LABEL will contain the PDS label
;							(Will only be valid if /STRIP_PDS_HEADER is specified)
;
; RETURNS:
;   a string array containing the imported text    or [''] on error
;
; RESTRICTIONS:
;   None
;
; EXAMPLE:
;   text = p_read_text_file('c:\temp\hello_world.txt')
;
; MODIFICATION HISTORY:
;   Written by:  Stubbe F. Hviid  20061002

FUNCTION p_read_text_file, file, STRIP_PDS_HEADER=STRIP_PDS_HEADER, LABEL=LABEL

	if n_elements(file) eq 0 then file = dialog_pickfile(FILTER='*.*')

	on_ioerror, ioe

	openr, unit, file, /GET_LUN

	st = fstat(unit)

	if st.size eq 0 then begin
		close, unit
		free_lun, unit
		return, ['']
	endif

	fb = bytarr(st.size)
	readu, unit, fb
	free_lun, unit

	; count lines
	lines = where(fb eq 10b)
	line_count = n_elements(lines)

	if line_count eq 0 then return, ['']

	; open file again
	openr, unit, file, /GET_LUN

	out = strarr(line_count)

	readf, unit, out

	if eof(unit) eq 0 then begin
		s = ''
		readf, unit, s
		out = [out, s]
	endif
	free_lun, unit

	if keyword_set(STRIP_PDS_HEADER) then begin

		; first check if a label is present
		end_found = 0
		type_found = 0

		for i=0, n_elements(out)-1 do begin
			t = strtrim(out[i], 2)
			if t eq 'END' then end_found = 1
			if strmid(t, 0, 11) eq 'RECORD_TYPE' then type_found = 1
		endfor

		; if no label was found the simply return
		if end_found eq 0 then begin
			print, 'No PDS label found in file: ' + file
			LABEL = ['']
			return, out
		endif

		; Ok a label was found, then strip it off the text
		LABEL = ['']
		for i=0,n_elements(out)-1 do begin
			t = strtrim(out[i], 2)

			if t eq 'END' then begin
				LABEL = [LABEL[1:*], out[i]]
				if i+1 ge n_elements(out) then return, [''] else return, out[i+1:*]
			endif else begin
				LABEL = [LABEL, out[i]]
			endelse

		endfor
	endif else return, out

	return, ['']

	ioe:

	return, ['']
END