; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;	Unauthorized reproduction prohibited.
;+
; NAME:
;	PRO p_setvalue, header, tag, value, unit
;
; PURPOSE:
;	Change or add a header tag to a PDS header
;
; CALLING SEQUENCE:
;	 p_setvalue, header, tag, value, unit
;
; INPUTS:
;	header:	Valid PDS header structure of file being read
;   tag:    Tag path of the item to add/modify
;   value:  the value to be written
;
; OPTIONAL INPUT:
;   unit:   The physical unit of the item to be written
;
; KEYWORD PARAMETERS:
;	GROUP		if set the function assumes that the tag refers to a group object
;		    	if group is not specified then all new tag clusters created will be OBJECTS
;   LABEL:  	Write string as PDS label string
;   NO_QUOTES:	Write as "no quote" label string    (required /LABEL)
;   FORMAT: Std text formating code
;   HISTORY:	Write the label to the history label not the main label
;
; EXAMPLE:
;	h = p_newhead()
;	p_setvalue, h, 'THIS', 'IS A TEST'		    ; add tag "HELLO' to header as string object with value 'IS A TEST'
;	p_setvalue, h, 'HELLO', 'WORLD', /LABEL		; add tag "HELLO' to header as label object with value 'WORLD'
;	p_setvalue, h, 'FLOAT_VALUE', 123.5		    ; add tag "FLOAT_VALUE' to header as float object with value 123.5
;	p_setvalue, h, 'FLOAT_ARRAY', [1.0, 1.5, 2.0]   ; add tag "FLOAT_ARRAY' to header as float array object with values [1.0, 1.5, 2.0]
;	p_setvalue, h, 'STR_ARRAY', ['Hello', 'World']   ; add tag "STR_ARRAY' to header as string array object with values ['Hello', 'World']
;
; MODIFICATION HISTORY:
; 	Written by:	Stubbe F. Hviid, 12/05-2004
;-

PRO p_setvalue, header, tag, ivalue, unit, GROUP=GROUP, HISTORY=HISTORY, _EXTRA=_EXTRA

	if keyword_set(HISTORY) then begin
		p_ensure_history, header
		history = header.history
		if strmid(tag, 0, 8)  ne 'HISTORY.' then rtag = 'HISTORY.' + tag else rtag = tag
		p_setvalue, history, rtag, ivalue, unit, GROUP=GROUP, _EXTRA=_EXTRA
		header = p_set_struct_tag(header, 'history', history)
		return
	endif

	; first handle structures seperately
	if size(ivalue, /TYPE) eq 8 then begin
		p_set_struct, header, tag, ivalue, GROUP=GROUP, _EXTRA=_EXTRA
		return
	endif

	gid = 'O'
	if keyword_set(GROUP) then gid = 'G'

	; if the tag already exists then simply modify existing tag
	t=p_value(header, tag, FOUND=FOUND, INDEX=idx)
	if FOUND ne 0 then begin
	  if size(t, /type) eq 8 then begin
	    ; the existing tag was a structure but the input value is a scalar
	    print, 'ERROR: You are replacing a structure (GROUP or OBJECT) with a scaler'
	    print, '       Not supported in single operation - first use p_delete_group + p_setvalue'
	    return
	  endif
		header.values[idx] = p_format_pds_value(ivalue, unit, _EXTRA=_EXTRA)
		return
	endif

	; get PDS formated value
	value = p_format_pds_value(ivalue, unit, _EXTRA=_EXTRA)

	; here make a special handling of pointer objects
	; Here we will attempt to place pointer objects within the 'POINTERS TO DATA OBJECTS' block of the header
	if strmid(tag, 0, 1) eq '^' then begin
		for i=0, n_elements(header.tags)-1 do begin
			if header.tags[i] eq '' AND strpos(header.values[i], 'POINTERS') gt 0 then begin
				p_add_tag, header, tag, value, INSERT_INDEX=i+1
				return
			endif
		endfor
	endif

	; tag not found then we need to create a new label item
	if strpos(tag, '.') lt 0 then begin
		; first handle the case where not groups are involved (system level labels)
		p_add_tag, header, tag, value
		return
	endif

	; it is a group or object
	; parse out the tag path
  p_parse_tag_path, tag, nodes, name

  start_index = n_elements(header.tags)-1;
  start_level = 0;
  for i=n_elements(nodes)-1, 0, -1 do begin
    if p_find_group_range(header, nodes[0:i], istart, iend) ne 0 then begin
      start_index = iend
      start_level = i
      found = 1
      break
    endif
  endfor

  if found eq 0 then begin
    for i=0, n_elements(nodes)-1 do p_add_tag, header, gid + '>', nodes[i]
    p_add_tag, header, name, value
    for i=n_elements(nodes)-1, 0, -1 do p_add_tag, header, gid + '<', nodes[i]
  endif else begin
    insert_index = start_index
    if start_level lt n_elements(nodes)-1 then begin
      for i=start_level+1, n_elements(nodes)-1 do p_add_tag, header, gid + '>', nodes[i], INSERT_INDEX=insert_index++
    endif
    p_add_tag, header, name, value, INSERT_INDEX=insert_index++
    if start_level lt n_elements(nodes)-1 then begin
      for i=n_elements(nodes)-1, start_level+1, -1 do p_add_tag, header, gid + '<', nodes[i], INSERT_INDEX=insert_index++
    endif
  endelse
END