; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;	Unauthorized reproduction prohibited.
;+
; NAME:
;	PRO p_rvicar_img_with_bh, header, image, OK=OK, BIN_HEADER=BIN_HEADER, PREFIX=PREFIX
;
; PURPOSE:
;	Read VICAR image from disk with VICAR images containing binary header and or prefix data
;
; CALLING SEQUENCE:
;	 p_read, file, header, image, OK=OK, BIN_HEADER=BIN_HEADER, PREFIX=PREFIX
;
; INPUTS:
;   file:   The file to read from
;
; OUTPUT:
;	file:	the filename opf the VICAR file to be read
;	header:	the output PDS header
;   image:  the output image
;
; KEYWORD PARAMETERS:
;	OK				will be 1 on success else 0
;   BIN_HEADER		will output the extracted binary header
;	PREFIX=PREFIX	will output the extracted binary prefix
;
; MODIFICATION HISTORY:
; 	Written by:	Stubbe F. Hviid, 12/05-2004
;-

PRO p_rvicar_img_with_bh, file, header, image, OK=OK, BIN_HEADER=BIN_HEADER, PREFIX=PREFIX

	; read header
	p_vicar_rhead, file, header, OK=OK, /RETAIN_VICAR_LABELS

	if OK eq 0 then return

	; calculate data offset
	pimage = strtrim(p_value(header, '^IMAGE', /AS_STRING), 2)
	data_pointer = long(pimage)
	img_offset = long(p_value(header, 'RECORD_BYTES')) * (data_pointer - 1)

	; extract dimensions
	w = long(p_value(header, 'IMAGE.LINE_SAMPLES'))
	h = long(p_value(header, 'IMAGE.LINES'))
	d = p_value(header, 'IMAGE.BANDS')
	if d eq '' then d = 1 else d = long(d)
	bd = long(p_value(header, 'IMAGE.SAMPLE_BITS'))

	if w eq 0 OR h eq 0 then begin
		print, 'no image data found'
		OK = 0
		return
	endif


	; set error handler
	on_ioerror, handle_error

	openr, unit, file, /GET_LUN

	;point_lun, unit, img_offset
	if img_offset gt 0 then begin
		tmp = bytarr(img_offset)
		readu, unit, tmp
	endif


	; generate image buffer
	type = p_value(header, 'IMAGE.SAMPLE_TYPE')

	if strpos(type, 'INTEGER') ge 0 then begin
		if strpos(type, 'UNSIGNED') ge 0 then begin
			if bd eq 8 then image = bytarr(w,h,d)
			if bd eq 16 then image = uintarr(w,h,d)
			if bd eq 32 then image = ulonarr(w,h,d)
		endif else begin
			if bd eq 8 then image = bytarr(w,h,d)
			if bd eq 16 then image = intarr(w,h,d)
			if bd eq 32 then image = lonarr(w,h,d)
		endelse
	endif else if type eq 'PC_REAL' then begin
		if bd eq 32 then image = fltarr(w,h,d)
		if bd eq 64 then image = dblarr(w,h,d)
	endif else if type eq 'IEEE_REAL' then begin
		if bd eq 32 then image = fltarr(w,h,d)
		if bd eq 64 then image = dblarr(w,h,d)
	endif else begin
		print, 'Unknown Datatype'
		OK=0
		return
	endelse

	nlb = fix(p_value(header, 'NLB'))
	nbb = fix(p_value(header, 'NBB'))
	recsize = fix(p_value(header, 'RECSIZE'))
	if recsize eq 0 then recsize = nbb + w * bd/8

	; read bin header
	if nlb ne 0 then begin
		BIN_HEADER = bytarr(nlb * recsize)
		readu, unit, BIN_HEADER
	endif

	PREFIX = bytarr(nbb, h)
	pfl = bytarr(nbb)
	il = reform(image[*, 0])

	for y=0L, h-1 do begin
		readu, unit, pfl
		readu, unit, il

		PREFIX[*, y] = pfl[*]
		image[*, y] = il[*]
	endfor

	; handle byte order
	if strpos(type, 'INTEGER') ge 0 then begin
		if strpos(type, 'LSB') ge 0 then begin
			if bd gt 8 then byteorder, image, /SWAP_IF_BIG_ENDIAN
			if bd eq 8 AND strpos(type, 'UNSIGNED') ge 0 then begin
				image = fix(image) + 127
			endif
		endif else begin
			if bd gt 8 then byteorder, image, /SWAP_IF_LITTLE_ENDIAN
			if bd eq 8 AND strpos(type, 'UNSIGNED') ge 0 then begin
				image = fix(image) + 127
			endif
		endelse
	endif else if type eq 'PC_REAL' then begin
		if bd eq 64 then begin
			byteorder, image, /SWAP_IF_BIG_ENDIAN, /L64SWAP
		endif else begin
			byteorder, image, /SWAP_IF_BIG_ENDIAN
		endelse
	endif else if type eq 'IEEE_REAL' then begin
		if bd eq 64 then begin
			byteorder, image, /SWAP_IF_LITTLE_ENDIAN, /L64SWAP
		endif else begin
			byteorder, image, /SWAP_IF_LITTLE_ENDIAN
		endelse
	endif

	close, unit
	free_lun, unit

	if n_elements(image) ne 0 then image = reform(image)

	OK=1
	return

	handle_error:
	print, 'Error reading image data'
	OK=0


END