; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
; Unauthorized reproduction prohibited.
;+
; NAME:
; FUNCTION p_read_shm_object, header
;
; PURPOSE:
; Extract shutter pulses out of Osiris PDS file
;
; CALLING SEQUENCE:
;  Result = p_read_shm_object(header)
;
; INPUTS:
; header: Valid PDS header structure of file being read
;
; KEYWORD PARAMETERS:
; OK:      will be 1 on success else 0
; VERBOSE: set to enable all progress print statements (only errors will be shown)
;
; RETURNS:
; on success: structure containing the shutter data
;      {type: 'shm', b1: <blade 1 pulses>, b2: <blade 2 pulses>,
;           time1: <time blade 1>, vel1: <velocity blade 1>,
;           time2: <time blade 2>, vel2: <velocity blade 2>}
; on failure: -1
;
; MODIFICATION HISTORY:
;   Written by: Stubbe F. Hviid, 12/05-2004
;-

PRO osi_pulse2velocity, pulse_array, out_time, out_velo

  if n_elements(pulse_array) lt 10 then begin
    out_time = [0.0]
    out_velo = [0.0]
    return
  endif


  pulse_count = n_elements(pulse_array)

    fAnglePerStripe = !PI / 3600.00;
    PULSE_RESOLUTION = (1.0/2.1e6)

  time_array = fltarr(1);
  velo_array = fltarr(1);

  for i=1.0,pulse_count-2 do begin

    time = ((pulse_array[i+1] + pulse_array[i-1]) / 2.0) * PULSE_RESOLUTION;

        delta_time = ((pulse_array[i+1] - pulse_array[i-1]) / 2.0) * PULSE_RESOLUTION;

        fRot = float(i) * fAnglePerStripe;
        if delta_time ne 0.0 then begin
          fDeltaRotVel = fAnglePerStripe/delta_time;
          velo =  0.1*(cos (fRot  - 0.07854)) * fDeltaRotVel;

      time_array = [time_array, time]
      velo_array = [velo_array, velo]
    endif
  endfor

  if n_elements(time_array) gt 1 then out_time = time_array[1:n_elements(time_array)-1]
  if n_elements(velo_array) gt 1 then out_velo = velo_array[1:n_elements(velo_array)-1]

  if n_elements(out_time) gt 0 then begin
    index = where(out_time lt 0.140)
    out_time = out_time[index]
    out_velo = out_velo[index]
  endif

END


FUNCTION p_read_shm_object, header, OK=OK, VERBOSE=VERBOSE

  on_ioerror, handle_error

  openr, unit, header.filename, /GET_LUN

  ; sewt default
  blade1 = [-1]
  blade2 = [-1]
  OK = 0

  ; try to read blade 1 pulses
  pb1 = p_value(header, '^BLADE1_PULSE_ARRAY')
  if pb1 ne '' then begin
    count = p_value(header, 'BLADE1_PULSE_ARRAY.AXIS_ITEMS')

    if keyword_set(VERBOSE) then print, 'Found ' + strtrim(count[0], 2) + ' Blade 1 pulses'

    blade1 = ulonarr(count[0])
    point_lun, unit, (long(pb1)-1) * long(p_value(header, 'RECORD_BYTES'))
    readu, unit, blade1
    byteorder, blade1, /SWAP_IF_BIG_ENDIAN
    OK = 1
  endif else begin
    blade1 = p_value(header, 'SHUTTER_PULSES.B1_PULSES')
    if string(blade1[0]) ne '' then OK = 1
  endelse

  ; try to read blade 2 pulses
  pb2 = p_value(header, '^BLADE2_PULSE_ARRAY')
  if pb2 ne '' then begin
    count = p_value(header, 'BLADE2_PULSE_ARRAY.AXIS_ITEMS')

    if keyword_set(VERBOSE) then print, 'Found ' + strtrim(count[0], 2) + ' Blade 2 pulses'

    blade2 = ulonarr(count[0])
    point_lun, unit, (long(pb2)-1) * long(p_value(header, 'RECORD_BYTES'))
    readu, unit, blade2
    byteorder, blade2, /SWAP_IF_BIG_ENDIAN
    OK = 1
  endif else begin
    blade2 = p_value(header, 'SHUTTER_PULSES.B2_PULSES')
    if string(blade2[0]) ne '' then OK = 1
  endelse

  if blade1[0] ne -1 then begin
    osi_pulse2velocity, blade1, time1, vel1
  endif else begin
    time1 = [-1.0]
    vel1 = [-1.0]
  endelse

  if blade2[0] ne -1 then begin
    osi_pulse2velocity, blade2, time2, vel2
  endif else begin
    time2 = [-1.0]
    vel2 = [-1.0]
  endelse

  ; close file
  close, unit
  free_lun, unit

  ; build output structure
  OK = 1
  return, {type: 'shm', b1: blade1, b2: blade2, time1: time1, vel1: vel1, time2: time2, vel2: vel2}

  handle_error:
  OK = 0
  return, -1
END