; Copyright (c) 2010, Stubbe F. Hviid.  All rights reserved.
;   Unauthorized reproduction prohibited.
;+
; NAME:
;	FUNCTION p_read_csv, file, DISCARD_LINES=DISCARD_LINES, NAMES=NAMES
;
; PURPOSE:
;	Read in tabular text data from ascii tables or CSV type files
;   ' ', ';', ',', '\t' is supported at field separator
;   The function will automatically detect the presense of column name and description rows in the input text
;
; CATEGORY:
;	Data import
;
; CALLING SEQUENCE:
; 	Result = p_read_csv(file)
;
; INPUTS:
;	file:	the filename of the table to be read
;
; KEYWORD PARAMETERS:
;	DISCARD_LINES:		specify to skip number of lines
;
; OUTPUTS:
;	Result will be a data structure with the form:
;          {type: 'csvdata', filename: file, columns: names, descriptions: descriptions, datatypes: datatypes, data: data}
;
;
; RESTRICTIONS:
;	The input file must be in ascii format with the row fields separated by either ' ', ';', ',' or TAB
;
; EXAMPLE:
;	data = p_read_csv(pickfile())	; read data from file with output to data and the column names to names
;	data = p_read_csv(pickfile(), DISCARD_LINES=10) ; discard the first 10 lines of the file
;
; MODIFICATION HISTORY:
; 	Written by:	Stubbe F. Hviid   Nov 2006
;-

FUNCTION p_read_csv, file, DISCARD_LINES=DISCARD_LINES

	; read text
	text = p_read_text_file(file);

	; strip comment lines
	tmp = strmid(text, 0, 1)
	s = where(tmp ne '#')
	if s[0] ge 0 then text = text[s]


	; strip empty lines
	s = where(text ne '')
	if s[0] ne -1 then text = text[s] else return, -1

	; strip comment lines (lines starting with '#'
	s = where( strmid(strtrim(text,2), 0, 1) ne '#' )
	if s[0] ne -1 then text = text[s] else return, -1

	; strip header lines if requested
	if n_elements(DISCARD_LINES) ne 0 then text = text[DISCARD_LINES:*]


	; auto detect seperator character
	sep_char = ' '
	if strpos(text[0], ';') gt 0 then sep_char = ';'
	if strpos(text[0], STRING(9B)) gt 0 then sep_char = STRING(9B)
	if strpos(text[0], ',') gt 0 then sep_char = ','


	; count number of colums in input files
	cols = strtrim(STRSPLIT(text[0], sep_char, /EXTRACT), 2)

	names = strarr(n_elements(cols))
	for j=0, n_elements(cols)-1 do names[j] = 'COL' + strtrim(j, 2)
	descriptions = strarr(n_elements(cols))
	datatypes = bytarr(n_elements(cols))

	; auto detect if the first line contains column headers
	ch = byte(strmid(cols[0], 0, 1))
	is_data = 0
	if ch ge byte('0') AND ch le byte('9') then is_data = 1
	if ch eq byte('+') OR ch eq byte('-') then is_data = 1
	if ch eq byte('.') then is_data = 1
	if ch eq byte('"') then is_data = 1

	if is_data eq 0 then begin
		names = strtrim(cols, 2)
		text = text[1:*]
		cols = strtrim(STRSPLIT(text[0], sep_char, /EXTRACT), 2)
	endif

	; auto detect if the second line contains units (or description)
	if n_elements(cols) ge 1 AND is_data eq 0 then begin
		ch = byte(strmid(cols[0], 0, 1))
		is_data = 0
		if ch ge byte('0') AND ch le byte('9') then is_data = 1
		if ch eq byte('+') OR ch eq byte('-') then is_data = 1
		if ch eq byte('.') then is_data = 1
		if ch eq byte('"') then is_data = 1

		if is_data eq 0 then begin
			descriptions = strtrim(cols, 2)
			text = text[1:*]
		endif
	endif


	; define output matrix
	ncol = n_elements(cols)
	nrow = n_elements(text)

	; init matrix
	data = strarr(ncol, nrow)

	; read data
	for i=0,nrow-1 do begin
		cols = STRSPLIT(text[i], sep_char, /EXTRACT)
		for j=0, ncol-1 do begin
			tt = strtrim(cols[j], 2)
			ch = strmid(tt, 0, 1)

			if ch eq '"' OR ch eq "'" then begin
				data[j,i] = strmid(tt, 1, strlen(tt)-2)
				datatypes[j] = 7
			endif else begin
				data[j,i] = tt
				if strpos(tt, '.') ge 0 then datatypes[j] = 5 else datatypes[j] = 3
			endelse

		endfor
	endfor

	return, {type: 'csvdata', filename: file, columns: names, descriptions: descriptions, datatypes: datatypes, data: data}


END