; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;   Unauthorized reproduction prohibited.
;+
; NAME:
;   PRO p_write
;
; PURPOSE:
;   Write a PDS file using a PDS label header and an image
;
; CALLING SEQUENCE:
;   p_write, file, header, image
;
; INPUTS:
;   file  : The output filename
;   header:   Valid PDS header structure
;   image:  any 1,2,3 dim image (byte, int, uint, long, ulong, float, dbl) valid
;
; OPTIONAL INPUTS:
;   slbl1:      string with object name of secondare image object
;   simg1:      image to be save as secondary image object 'slbl1'
;   slbl2:      string with object name of secondare image object
;   simg2:      image to be save as secondary image object 'slbl2'
;   slbl3:      string with object name of secondare image object
;   simg3:      image to be save as secondary image object 'slbl3'
;   slbl4:      string with object name of secondare image object
;   simg4:      image to be save as secondary image object 'slbl4'
;   slbl5:      string with object name of secondare image object
;   simg5:      image to be save as secondary image object 'slbl5'
;   slbl6:      string with object name of secondare image object
;   simg6:      image to be save as secondary image object 'slbl6'
;
;   Note that sub image object must always come in pairs (if slbl# is specified the simg# must also be specified)
;
; KEYWORD PARAMETERS:
;   QUIET:				If set not progress message will be printed
;   VERSION_CONTROL:	Save image using version control
;					    (Existing image will be archived in ./PDS_Backup
;                        and a _V# element will be added to the filename of the existing
;						 file plus the PRODUCT_VERSION_ID label in the PDS header will be incremented)
;   RECORD_BYTES  The records block size to use for the save PDS file (default 512 bytes)
;                 (All records will be aligned to a RECORD_BYTES boundry)
;   IMAGE_HEADER: optional keyword for specifying an image header object
;                 The value specified must be a structure with the form:
;                 structure = {type: <image header type
;                              format: <either ASCII or BINARY>    - optional member default: ASCII
;                              header: <byte (BINARY) or string (ASCII) array with header data>
;                              description: <Extrenal file reference descriping the header> - optional input
;                             }
;	HEADER_OUT=header	Will contain the updated header struture used to write the image data to disk.
;						<header_in> is unmodified but the write process makes some changes to a copy of the header
;						before actually writing toi disk <HEADER_OUT> can be used to get this modified header
; SKIP_HISTORY:      If set the history object attachement will not be written (even if available)
; OUTPUTS:
;   NONE
;
; EXAMPLE:
;   p_read, pickfile(), h, d  ; read PDS file
;   pwrite, pickfile(), h, dist(200,200)
;
; MODIFICATION HISTORY:
;   Written by:  Stubbe F. Hviid, 12/05-2004
;-

FUNCTION p_handle_version_control, filename

	on_ioerror, ioerr

	; test for new file - if new then return version 1
	if file_test(filename) eq 0 then return, 1

	; at this point an existing file was found then backup the existing file

	; find directory of filename
	fd = p_parse_file_path(filename)
	backup_dir = fd.dir + '/PDS_Backup'


	; test is the backup directory exists - and if not create it

	if file_test(backup_dir) eq 0 then file_mkdir, backup_dir

	; read header of existing file
	p_rhead, filename, h, /HEADER_ONLY

	; extract version number from header
	version = fix(p_value(h, 'PRODUCT_VERSION_ID'))

	; move existing file to backup dir
	file_move, filename, backup_dir + '/' + fd.name + '_V' + strtrim(version, 2) + fd.ext, /OVERWRITE

	return, version + 1

	ioerr:
	return, 1
END

PRO p_add_image_object, header, name, image
    ; check for endian formal of computer
    no = 1
    byteorder, no, /SWAP_IF_LITTLE_ENDIAN
    if no ne 1 then endian = 'little' else endian = 'big'

    ; determine size of image
    t = size(image, /TYPE)
    dim = size(image, /DIM)

    w=1
    h=1
    d=1

    if n_elements(dim) ge 1 then w = dim[0]
    if n_elements(dim) ge 2 then h = dim[1]
    if n_elements(dim) ge 3 then d = dim[2]

    if w le 0 then w=1
    if h le 0 then h=1
    if d le 0 then d=1

    ; modify image object section
    p_setvalue, header, name + '.INTERCHANGE_FORMAT', 'BINARY'
    p_setvalue, header, name + '.LINE_SAMPLES', w
    p_setvalue, header, name + '.LINES', h
    p_setvalue, header, name + '.BANDS', d

    ; determine datatype
    bd=0
    fp=0

    if t eq 1 then bd=8
    if t eq 2 then bd=16
    if t eq 3 then bd=32
    if t eq 4 then begin
       bd=32
       fp=1
    endif
    if t eq 5 then begin
       bd=64
       fp=1
    endif

    if t eq 12 then bd=16
    if t eq 13 then bd=32
    if t eq 14 then bd=64
    if t eq 15 then bd=64

    if endian eq 'little' then begin
       st='LSB_'
       if t eq 4 OR t eq 5 then st='PC_'
    endif else begin
       st='MSB_'
       if t eq 4 OR t eq 5 then st='IEEE_'
    endelse

    if t lt 4 AND t ne 1 then begin
       st = st + 'INTEGER'
    endif else if t ge 4 AND t le 5 then begin
       st = st + 'REAL'
    endif else begin
       st = st + 'UNSIGNED_INTEGER'
    endelse

    p_setvalue, header, name + '.SAMPLE_TYPE', st, /LABEL, /NO_QUOTES
    p_setvalue, header, name + '.SAMPLE_BITS', bd, /LABEL

    miv = min(image, MAX=mav)

    p_setvalue, header, name + '.DERIVED_MINIMUM', double(miv)
    p_setvalue, header, name + '.DERIVED_MAXIMUM', double(mav)
    p_setvalue, header, name + '.MEAN', mean(image)

    p_setvalue, header, name + '.SAMPLE_BIT_MASK', 'NULL', /LABEL, /NO_QUOTES

END

FUNCTION p_calc_strarr_size, sa

  ; calc sizeof string array in bytes
  s = 0;
  for i=0, n_elements(sa)-1 do s += strlen(sa[i]) + 2

  ; return the size of the image in records
  return, s
END

FUNCTION p_size_image, image

  ; size in bytes of the various IDl data types
  ets = [0,1,2,4,4,8,0,0,0,0,0,0,2,4,8,8]

  ; get number of elements in image
  ec = n_elements(image)

  ; calc size in bytes of image
  is = ec * ets[size(image, /type)]

  ; chek for invalid data type (like string, struct, ...)
  if is eq 0 then return, 0

  ; return the size of the image in records
  return, is
END

PRO p_align_file_to_record, unit, expected_write, object_size_in_records, record_bytes

	object_size_in_bytes = object_size_in_records * record_bytes

	needed_write_pointer = expected_write + object_size_in_bytes

	stat = fstat(unit)

	if stat.size lt needed_write_pointer then begin
		ps = needed_write_pointer - stat.size
		padding = bytarr(ps)
		padding[*] = 0
		writeu, unit, padding
	endif

	expected_write += object_size_in_bytes
END

FUNCTION p_calc_record_count, size_in_bytes, record_bytes

  records = size_in_bytes

  for i=0, n_elements(size_in_bytes)-1 do begin
    records[i] = size_in_bytes[i] / record_bytes

    if (size_in_bytes[i] MOD record_bytes) ne 0 then (records[i])++
  endfor
  return, records
END

FUNCTION p_add_imageheader_object, header, image_header, output_binary_block

  ; make sure that the input image header is a struture
  if size(image_header, /type) ne 8 then return, 0

  ; parse input
  type = image_header.type
  if p_struct_tag_exists(image_header, 'format') ne 0 then format = strupcase(image_header.format)
  if p_struct_tag_exists(image_header, 'description') ne 0 then description = image_header.description
  data = image_header.header

  ; handle defaults
  if n_elements(format) eq 0 then format = 'ASCII'

  if format ne 'ASCII' then begin
    if size(data, /type) ne 1 then begin
      print, 'ILLEGAL image header data type - must be bytarr() for BINARY headers'
      return, 0
    endif
  endif else begin
    if size(data, /type) ne 7 then begin
      print, 'ILLEGAL image header data type - must be strarr() for ASCII headers'
      return, 0
    endif
  endelse

  ; calc size
  if format eq 'ASCII' then begin
    hs = p_calc_strarr_size(data)
    output_binary_block = [byte(data[0]), byte('0D'X), byte('0A'X)]
    if n_elements(data) gt 1 then begin
      for i=1, n_elements(data)-1 do begin
        output_binary_block = [output_binary_block, byte(data[i]), byte('0D'X), byte('0A'X)]
      endfor
    endif
  endif else begin
    hs = n_elements(data)
    output_binary_block = data
  endelse


  p_setvalue, header, 'IMAGE_HEADER.HEADER_TYPE', type
  p_setvalue, header, 'IMAGE_HEADER.INTERCHANGE_FORMAT', format
  p_setvalue, header, 'IMAGE_HEADER.BYTES', hs

  if n_elements(description) ne 0 then p_setvalue, header, 'IMAGE_HEADER.^DESCRIPTION", description

  return, 1
END


PRO p_write, filename, header_in, image_in, slbl1, simg1, slbl2, simg2, slbl3, simg3, slbl4, simg4, slbl5, simg5, slbl6, simg6, $
			 QUIET=QUIET, $
			 VERSION_CONTROL=VERSION_CONTROL, $
			 RECORD_BYTES=RECORD_BYTES, $
			 SKIP_HISTORY=SKIP_HISTORY, $
			 IMAGE_HEADER=IMAGE_HEADER, $
			 HEADER_OUT=HEADER_OUT

  on_ioerror, ioerr

	if NOT keyword_set(QUIET) then print, 'Saving PDS file: ' + filename
	if n_elements(RECORD_BYTES) eq 0 then RECORD_BYTES = 512

  ; make local copy of input
  header = header_in
  image = image_in

	; First handle version control
	if keyword_set(VERSION_CONTROL) then begin
		version = p_handle_version_control(filename)
		p_setvalue, header, 'PRODUCT_VERSION_ID', strtrim(version, 2)
	endif

	if p_value(header, 'PDS_VERSION_ID', FOUND=found) ne 'PDS3' then begin
		if found eq 0 then begin
			header = p_set_struct_tag(header, 'tags', ['PDS_VERSION_ID', header.tags])
			header = p_set_struct_tag(header, 'values', ['PDS3', header.values])
		endif else p_setvalue, header, 'PDS3'
	endif


  ; clear existing binary objects from the header
  objs = p_find_embedded_objects(header)

  for i=1, n_elements(objs)-1 do begin
    if objs[i] ne 'IMAGE' AND objs[i] ne 'HISTORY' then begin
      p_delete_group, header, objs[i], /OBJECT_ONLY
      p_delete_tag, header, '^' + objs[i]
    endif
  endfor

    ; prelim init of header
    p_setvalue, header, 'RECORD_TYPE', 'FIXED_LENGTH'
    p_setvalue, header, 'RECORD_BYTES', 512, /LABEL
    p_setvalue, header, 'FILE_RECORDS', 'XXXXXXXXXXXXX', /LABEL
    p_setvalue, header, 'LABEL_RECORDS', 'XXXXXXXXXXXXX', /LABEL

    fs = p_parse_file_path(filename)

    p_setvalue, header, 'FILE_NAME', fs.filename

    ; define preliminary pointer table
    if n_elements(image) ne 0 then p_setvalue, header, '^IMAGE', 'XXXXXXXXXXXXX'
    if p_struct_tag_exists(header, 'history') AND NOT keyword_set(SKIP_HISTORY) then begin
      if n_elements(slbl1) ne 0 then p_setvalue, header, '^HISTORY', 'XXXXXXXXXXXXX'
      history_text = p_format(header.history)
    endif else begin
      p_delete_tag, header, '^HISTORY'
    endelse


    if n_elements(slbl1) ne 0 then p_setvalue, header, '^'+slbl1, 'XXXXXXXXXXXXX'
    if n_elements(slbl2) ne 0 then p_setvalue, header, '^'+slbl2, 'XXXXXXXXXXXXX'
    if n_elements(slbl3) ne 0 then p_setvalue, header, '^'+slbl3, 'XXXXXXXXXXXXX'
    if n_elements(slbl4) ne 0 then p_setvalue, header, '^'+slbl4, 'XXXXXXXXXXXXX'
    if n_elements(slbl5) ne 0 then p_setvalue, header, '^'+slbl5, 'XXXXXXXXXXXXX'
    if n_elements(slbl6) ne 0 then p_setvalue, header, '^'+slbl6, 'XXXXXXXXXXXXX'

    if n_elements(IMAGE_HEADER) ne 0 then p_setvalue, header, '^IMAGE_HEADER', 'XXXXXXXXXXXXX'


    ; define image object structures
    if n_elements(image) ne 0 then p_add_image_object, header, 'IMAGE', image
    if n_elements(slbl1) ne 0 then p_add_image_object, header, slbl1, simg1
    if n_elements(slbl2) ne 0 then p_add_image_object, header, slbl2, simg2
    if n_elements(slbl3) ne 0 then p_add_image_object, header, slbl3, simg3
    if n_elements(slbl4) ne 0 then p_add_image_object, header, slbl4, simg4
    if n_elements(slbl5) ne 0 then p_add_image_object, header, slbl5, simg5
    if n_elements(slbl6) ne 0 then p_add_image_object, header, slbl6, simg6

    if n_elements(IMAGE_HEADER) ne 0 then begin
      if p_add_imageheader_object(header, IMAGE_HEADER, bin_image_header) eq 0 then begin
        print, 'Illegal Image header Object - Cannot save file'
        return
      endif
    endif



    ; generate preliminary header object
    lhdr = p_format(header)

    ; generate real pointer table
    ob_size   = [p_calc_strarr_size(lhdr)]
    if n_elements(history_text) ne 0 then ob_size = [ob_size, p_calc_strarr_size(history_text)]

    if n_elements(image) ne 0 then ob_size = [ob_size, p_size_image(image)]
    if n_elements(simg1) ne 0 then ob_size = [ob_size, p_size_image(simg1)]
    if n_elements(simg2) ne 0 then ob_size = [ob_size, p_size_image(simg2)]
    if n_elements(simg3) ne 0 then ob_size = [ob_size, p_size_image(simg3)]
    if n_elements(simg4) ne 0 then ob_size = [ob_size, p_size_image(simg4)]
    if n_elements(simg5) ne 0 then ob_size = [ob_size, p_size_image(simg5)]
    if n_elements(simg6) ne 0 then ob_size = [ob_size, p_size_image(simg6)]

    if n_elements(bin_image_header) ne 0 then ob_size = [ob_size, n_elements(bin_image_header)]

    for i=0,n_elements(ob_size)-1 do begin
      if ob_size[i] eq 0 then begin
         print, 'Incompatible segment detected (probably illegal image data type) - Error Saving file'
         return
      endif
    endfor


    ob_records = p_calc_record_count(ob_size, RECORD_BYTES)

    ; update file and label records
    p_setvalue, header, 'FILE_RECORDS', long(total(ob_records)), /LABEL
    p_setvalue, header, 'LABEL_RECORDS', ob_records[0], /LABEL

    ; updated header with correct offsets
    offset = ob_records[0]
    idx = 1
    if n_elements(history_text) ne 0 then begin
      p_setvalue, header, '^HISTORY', offset+1
      offset += ob_records[idx++]
    endif
    if n_elements(image) ne 0 then begin
      p_setvalue, header, '^IMAGE', offset+1
      offset += ob_records[idx++]
    endif
    if n_elements(slbl1) ne 0 then begin
      p_setvalue, header, '^' + slbl1, offset+1
      offset += ob_records[idx++]
    endif
    if n_elements(slbl2) ne 0 then begin
      p_setvalue, header, '^' + slbl2, offset+1
      offset += ob_records[idx++]
    endif
    if n_elements(slbl3) ne 0 then begin
      p_setvalue, header, '^' + slbl3, offset+1
      offset += ob_records[idx++]
    endif
    if n_elements(slbl4) ne 0 then begin
      p_setvalue, header, '^' + slbl4, offset+1
      offset += ob_records[idx++]
    endif
    if n_elements(slbl5) ne 0 then begin
      p_setvalue, header, '^' + slbl5, offset+1
      offset += ob_records[idx++]
    endif
    if n_elements(slbl6) ne 0 then begin
      p_setvalue, header, '^' + slbl6, offset+1
      offset += ob_records[idx++]
    endif

    if n_elements(slbl6) ne 0 then begin
      p_setvalue, header, '^' + slbl6, offset+1
      offset += ob_records[idx++]
    endif

    if n_elements(bin_image_header) ne 0 then begin
      p_setvalue, header, '^IMAGE_HEADER', offset+1
      offset += ob_records[idx++]
    endif

    ; generate the final header for export
    lhdr = p_format(header)

    ; finally save the file

    ; save the file
    openw, unit, filename, /GET_LUN

	; init expected write bytes
	expected_write = 0

    ; save the header
    for i=0, n_elements(lhdr)-1 do printf, unit, lhdr[i]
    p_align_file_to_record, unit, expected_write, ob_records[0], RECORD_BYTES
    idx = 1

    ; save history if existing
    if n_elements(history_text) ne 0 then begin
      for i=0, n_elements(history_text)-1 do begin
        printf, unit, history_text[i]
      endfor
      p_align_file_to_record, unit, expected_write, ob_records[idx++], RECORD_BYTES
    endif

    ; save images
    if n_elements(image) ne 0 then begin
      writeu, unit, image
      p_align_file_to_record, unit, expected_write, ob_records[idx++], RECORD_BYTES
    endif

    if n_elements(simg1) ne 0 then begin
      writeu, unit, simg1
      p_align_file_to_record, unit, expected_write, ob_records[idx++], RECORD_BYTES
    endif

    if n_elements(simg2) ne 0 then begin
      writeu, unit, simg2
      p_align_file_to_record, unit, expected_write, ob_records[idx++], RECORD_BYTES
    endif

    if n_elements(simg3) ne 0 then begin
      writeu, unit, simg3
      p_align_file_to_record, unit, expected_write, ob_records[idx++], RECORD_BYTES
    endif

    if n_elements(simg4) ne 0 then begin
      writeu, unit, simg4
      p_align_file_to_record, unit, expected_write, ob_records[idx++], RECORD_BYTES
    endif

    if n_elements(simg5) ne 0 then begin
      writeu, unit, simg5
      p_align_file_to_record, unit, expected_write, ob_records[idx++], RECORD_BYTES
    endif

    if n_elements(bin_image_header) ne 0 then begin
      writeu, unit, bin_image_header
      p_align_file_to_record, unit, expected_write, ob_records[idx++], RECORD_BYTES
    endif


    close, unit
    free_lun, unit

	; change the filename member of the header structure
	HEADER_OUT = p_set_struct_tag(header_in, 'filename', filename)

	return

	ioerr:
	print, 'IO error when saving ' + filename
END