/*
 * DFMS_FindAllPeaks_Class.cc
 *
 *  Created on: Aug 21, 2013
 *      Author: marinaldi
 */

#include "DFMS_FindAllPeaks_Class.hh"

DFMS_FindAllPeaks_Class::DFMS_FindAllPeaks_Class(double *c, int il, int ir, double m, double s) {

	index = 0;

	iXl = il;
	iXr = ir;

	maxPkLoc = 0;
	maxPkVal = 0;
	numGdPks = 0;
	maxId = 0;

	mean = m;
	sDevMean = s;

	peakLoc.clear();
	peakVal.clear();
	peakWd.clear();
	inBounds.clear();
	pkAboveThresh.clear();

    peakInd = new int[MAXNUMPEAKS];
    maxVal = new double[MAXNUMPEAKS];
    minVal = new double[MAXNUMPEAKS];
    numPeaks = 0;
    absopPeaks = new int[MAXNUMPEAKS];
    numAbsopPeaks = 0;

    count = new double[NUMPIX];
    for(int i=0; i<NUMPIX; ++i) {
        count[i] = c[i];
        //cout << "Y[" << i << "] = " << count[i] << endl;
    }
    //exit(0);

}

DFMS_FindAllPeaks_Class::~DFMS_FindAllPeaks_Class() {

	delete[] count; count = 0;
	delete[] peakInd; peakInd = 0;
	delete[] maxVal; maxVal=0;
	delete[] minVal; minVal=0;
	delete[] absopPeaks; absopPeaks=0;
	peakLoc.clear();
	peakVal.clear();
	peakWd.clear();
	inBounds.clear();
	pkAboveThresh.clear();
}

//
// ------------------------- detectAllPeaks ------------------------------------
//
//==============================================================================
// Method Description
//==============================================================================
// This routine finds peaks in the input data.  This is used to find
// verification peaks
//
//   variables set by method
//   maxPeaks, - maximum number of emission peaks
//   maxAbsopPeaks, - maximum number of absorption peaks
//   delta, - delta used for distinguishing peaks
//   emiFirst - should we search emission peak first of absorption peak first?

//   Output
//   peaks, - emission peaks will be put here
//   numPeaks, - number of emission peaks found
//   absopPeaks, - absorption peaks will be put here
//   numAbsopPeaks, - number of absorption peaks found
//
// returns:
//   1 for a successful run or -1 to indicate an error to be handled
// =============================================================================
// History: Originally inspired by Eli Billauer's peakdet for MATLAB
//          Converted to C (called detect_peak.c) by Hong Xu.
//          Adapted with slight mods by Mike Rinaldi, August 2013
// =============================================================================
int DFMS_FindAllPeaks_Class::detectAllPeaks()
{

	static int lowIndx=20;   // Don't look before lowIndx or past NUMPIX-lowIndx
    int     i;
    double  mx;
    double  mn;
    int     maxPos = lowIndx;
    int     minPos = lowIndx;

    int maxPeaks = MAXNUMPEAKS;
    int maxAbsopPeaks = MAXNUMPEAKS;
    double delta=sDevMean*(NUMSDEV+1);
    int emiFirst=1;

    int isDetectingEmi = emiFirst;

    mx = count[lowIndx];
    mn = count[lowIndx];

    numPeaks = 0;
    numAbsopPeaks = 0;

    for(i = lowIndx; i < NUMPIX-lowIndx; ++i) {
        if(count[i] > mx) {
            maxPos = i;
            mx = count[i];
        }
        if(count[i] < mn) {
            minPos = i;
            mn = count[i];
        }

        if(isDetectingEmi && count[i] < mx - delta) {

            if(numPeaks >= maxPeaks) return 1;

            peakInd[numPeaks] = maxPos+1;
            maxVal[numPeaks] = count[maxPos];
            //cout << "peaks[" << numPeaks << "]=" << count[maxPos] << " at maxPos=" << maxPos << endl;

            ++numPeaks;

            isDetectingEmi = 0;

            i = maxPos - 1;

            mn = count[maxPos];
            minPos = maxPos;

        } else if((!isDetectingEmi) && count[i] > mn + delta) {

            if(numAbsopPeaks >= maxAbsopPeaks) return 2;

            absopPeaks[numAbsopPeaks] = minPos+1;
            minVal[numAbsopPeaks] = count[minPos];
            //cout << "absopPeaks[" << *numAbsopPeaks << "]=" << count[minPos] << " at minPos=" << minPos << endl;

            ++numAbsopPeaks;

            isDetectingEmi = 1;

            i = minPos - 1;

            mx = count[minPos];
            maxPos = minPos;
        }

    }

    return 0;
}

//
// ---------------------- findPeaksAboveThreshold ------------------------------
//
//==============================================================================
// Method Description
//==============================================================================
// This routine finds peaks above the PKMINHGHT threshold.
//
//   Input
//	 int* peaks - The location of the peaks
//   int  numPeaks - Total number of peaks
//   int* maxVal  -  The Value at peak loc
//
//   Output
//   int* goodPeaks - Peaks above Threshold
//
// returns:
//   number of good peaks
// =============================================================================
// History: Written by Mike Rinaldi, March 2014
// =============================================================================
int DFMS_FindAllPeaks_Class::findPeaksAboveThreshold(double threshold, double meanOffset) {

	string sFunctionName = "DFMS_FindAllPeaks_Class::findPeaksAboveThreshold";

	// Find all the peaks
	int nSuccess = detectAllPeaks();

	for (int i=0; i<numPeaks; i++) {
		//cout << "maxVal[" << i << "] = " << maxVal[i] << "  peakInd[" << i << "] = " << peakInd[i] << endl;
		if (maxVal[i] >= threshold && maxVal[i] - meanOffset > PKHEIGHTLIMIT) {
			pkAboveThresh.push_back(true);
		} else {
			pkAboveThresh.push_back(false);
		}
		peakLoc.push_back(peakInd[i]);
		if (peakInd[i] >= iXl && peakInd[i] <= iXr) {
			inBounds.push_back(true);
		} else {
			inBounds.push_back(false);
		}
		peakVal.push_back(maxVal[i]);
		//double pWd = findWidth(peakInd[i], maxVal[i]);
		peakWd.push_back(9.0);
	}

	// Look for the maximum of all the found peaks
	findMaxPeak();

	numGdPks = peakLoc.size();

	if (numGdPks <= 0 || maxId == -1) {
		sInfoMessage = "No Peaks above threshold were found";
		writeToLog(sInfoMessage, sFunctionName, INFO);
		return 0;
	} else {
		return peakLoc.size();
	}

	return 0;
}

//
// ----------------------------- findMaxPeak -------------------------------------
//
//==============================================================================
// Method Description
//==============================================================================
// This routine finds the location and value of the tallest peak and stores that
// info into the peakInfo struct
//
// returns:
//   none
// =============================================================================
// History: Written by Mike Rinaldi, March 2014
// =============================================================================
void DFMS_FindAllPeaks_Class::findMaxPeak() {

	int idx = -1;
	double max =  -1.0e99;

	// find the tallest peak within Bounds
	for (int i=0; i<peakLoc.size(); i++) {
		//cout << "peakVal[" << i << "] = " << peakVal[i] << "  inbounds[" << i << "] = " << inBounds[i] << "  pkAboveThresh[" << i << "] = " <<pkAboveThresh[i] << endl;
		if (peakVal[i] > max && inBounds[i] && pkAboveThresh[i])
		{
			max = peakVal[i];
			idx = i;
		}
	}

	// Make sure peak is inBounds
	if (idx != -1) {
		maxPkLoc = peakLoc[idx];
		maxPkVal = peakVal[idx];
		if (inBounds[idx] && pkAboveThresh[idx]) {
			maxId = idx;
		} else {
			maxId = -1;
		}
	} else {
		maxId = -1;
	}

}

//
// ----------------------------- findWidth -------------------------------------
//
//==============================================================================
// Method Description
//==============================================================================
// This routine finds the peaks FWHM value
//
//   Input
//	 int peakLoc - The location of the peak
//   double pkHght - The Value at peak loc
//
// returns:
//   peak width (FWHM)
// =============================================================================
// History: Written by Mike Rinaldi, March 2014
// =============================================================================
double DFMS_FindAllPeaks_Class::findWidth(int peakLoc, double pkHght) {

	double halfMax = (pkHght-mean)/2.0;
	if (halfMax < mean) halfMax = pkHght/2.0;
	double lhmax=0.0;
	double rhmax=0.0;

	// Find the pixel associated with the half max point on
	// the left side of the peak
	for (int i=peakLoc; i>peakLoc-15; i--) {
		cout << "count[" << i << "] = " << count[i] << endl;
		if (count[i] > halfMax && count[i-1] <= halfMax) {
			lhmax = i;
			break;
		}
	}

	// Find the pixel associated with the half max point on
	// the right side of the peak
	for (int i=peakLoc; i<peakLoc+15; i++) {
		if (count[i] > halfMax && count[i+1] < halfMax) {
			rhmax = i;
			break;
		}
	}

	if (lhmax != 0 && rhmax != 0) {
		return double(rhmax - lhmax);
	}

	return 0;

}


