
#include "DFMS_MPST_Class.hh"

//
// ---------------------------- Input Constructor ------------------------------------
//
// This is the constructor form to use for reading an MPST file
//
DFMS_MPST_Class::DFMS_MPST_Class(string p, string f) {
     
	path = p;
	fileName = f;
	MPSTfile = path+fileName;
	MPSTrows = 0;
	MPSTcols = 0;
	creationTime = getProcessTime(1);

}

//
// ---------------------------- Output Constructor ------------------------------------
//
// This is the constructor form to use for writing an MPST file
//
DFMS_MPST_Class::DFMS_MPST_Class(string p, string f, int n, mapstr &P, 
                               mapstr &d, str2D &MPSTin) {

	path = p;
	fileName = f;
	MPSTfile = path+fileName;
	MPSTrows = n;
	MPSTcols = 0;
	creationTime = fileName.substr(9,15);
	sModeOut = fileName.substr(0,3);
	MPSTobj = d;	     // Copy input descObj map using overloaded map::operator=
	PDSheader = P ;	// Copy input PDSheader using overloaded map::operator=
     
	// Make a local copy of MPSTin
	for (int i=0; i<MPSTrows; ++i) {
		MPSTdata.push_back(sslice());  // Create one row
		for (int j=0; j<7; ++j) {
			MPSTdata[i].push_back(MPSTin[i][j]);
		}
	}



}

//
// ----------------------------- Destructor ------------------------------------
//
DFMS_MPST_Class::~DFMS_MPST_Class() {
    
	// Release dynamically allocated memory
	MPSTdata.clear();
}

//
// ---------------------------- Open PDS file ------------------------------------
//
int DFMS_MPST_Class::openFile4Out(string file, fstream &str) {

	string sFunctionName = "DFMS_MPST_Class::openFile4Out";

	// Open the file input stream for writing
	str.open(file.c_str(), fstream::out | fstream::binary);

	if(!str.is_open()) {
		sErrorMessage = "Error opening MPS File: "+file;
		sErrorMessage += " for writing";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	return 1;
}

//
// ----------------------------- writePDSheader ------------------------------------
//
int DFMS_MPST_Class::writePDSheader(fstream &os) {
    
    char tmp[81];
    string stmp;
    int lbl_recs = 41;
    stmp = "PDS_VERSION_ID                   =     PDS3";
    sprintf(tmp,"%-78s",stmp.c_str());
    os << string(tmp) << dfmsEOL;
    stmp = "LABEL_REVISION_NOTE              =     \"2012-08-22,Thierry Semon(UoB),";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "                                        version2.1 release\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "RECORD_TYPE                      =     FIXED_LENGTH";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "RECORD_BYTES                     =     80";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "FILE_RECORDS                     =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lbl_recs+MPSTrows)).c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "LABEL_RECORDS                    =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lbl_recs)).c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "^DFMS_MASS_PEAK_SEARCH_TABLE     =     42";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "DATA_SET_ID                      =     \"RO-A-ROSINA-3-AST2-V2.0\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "DATA_SET_NAME                    =     \"ROSETTA-ORBITER LUTETIA ROSINA 3";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "                                        AST2 V2.0\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
       // Modify the PDS Header - PRODUCT_ID
	string name = fileName.substr(0,5) + "_D_" +
                  fileName.substr(16,15);
    stmp = "PRODUCT_ID                       =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),name.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "PRODUCT_CREATION_TIME            =     ";
       string tTime = getProcessTime(1);
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),tTime.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "PRODUCT_TYPE                     =     RDR";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "PROCESSING_LEVEL_ID              =     \"3\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "MISSION_ID                       =     ROSETTA";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "MISSION_NAME                     =     \"INTERNATIONAL ROSETTA MISSION\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "INSTRUMENT_HOST_NAME             =     \"ROSETTA-ORBITER\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "INSTRUMENT_HOST_ID               =     RO";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "INSTRUMENT_NAME                  =     \"ROSETTA ORBITER SPECTROMETER FOR";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "                                        ION AND NEUTRAL ANALYSIS\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "INSTRUMENT_ID                    =     ROSINA";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "INSTRUMENT_MODE_ID               =     ";
       sprintf(tmp,"%-30s%-39s",stmp.c_str(),sModeOut.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "^INSTRUMENT_MODE_DESC            =     \"DFMS_MODE_DESC.ASC\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "INSTRUMENT_TYPE                  =     \"MASS SPECTROMETER\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "DETECTOR_ID                      =     DFMS";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "DETECTOR_DESC                    =     \"DOUBLE FOCUSING MASS SPECTROMETER\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "CHANNEL_ID                       =     MC";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "PRODUCER_ID                      =     ROSETTA_ROSINA";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "PRODUCER_FULL_NAME               =     \"KATHRIN ALTWEGG\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    stmp = "PRODUCER_INSTITUTION_NAME        =     \"UNIVERSITY OF BERN\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    // Add an 79 character blank line to separate PDSheader from first Object descriptor
	sprintf(tmp,"%-78s"," ");
	os << string(tmp) << dfmsEOL;
	
	return 1;

}

//
// ------------------------------- formMPSTobj ----------------------------------------
//
int DFMS_MPST_Class::formMPSTobj() {

	// Modify the Desc Object
	MPSTobj["ROWS"] = util_intToString(MPSTrows);

	return 1;

}

//
// ------------------------------- getModeID ------------------------------------
//
string DFMS_MPST_Class::getModeID() {

	return fileName.substr(0,5);

}

//
// ------------------------------ readMPSTdata ------------------------------------
//

// Assumes Mass Peak File has 7 data fields each followed by a comma (,)
// No assumption is made on field length 
int DFMS_MPST_Class::readMPSTdata(fstream &is) {

	string sFunctionName = "DFMS_MPST_Class::readMPSTdata";

	char input[100];
	int cpos0,cpos1,cpos2;
	int cpos3,cpos4,cpos5,cpos6;   // positions of commas within string/substring
	int len;                 // length of a string  

	// Now read the  MPST data into the data string array
 	for (int i=0; i<MPSTrows; i++) {

		// Use vector objects dynamic fill to add values
		MPSTdata.push_back(sslice());

		is.getline(input,100); 
		string line = string(input);
		// Peak Name - Look for 1st comma
		cpos0 = line.find_first_of(",",0);
		len = cpos0;
		MPSTdata[i].push_back(util_trim(line.substr(0,len)));
		// Peak Mass - Look for 2nd comma
		cpos1 = line.find_first_of(",",cpos0+1);
		len = cpos1-cpos0-1;
		MPSTdata[i].push_back(util_trim(line.substr(cpos0+1,len)));
 		// Peak Cal type - Look for 3rd comma
		cpos2 = line.find_first_of(",",cpos1+1);
		len = cpos2-cpos1-1;
		MPSTdata[i].push_back(util_trim(line.substr(cpos1+1,len)));
 		// Expected Peak bin Center - Look for 4th comma
		cpos3 = line.find_first_of(",",cpos2+1);
		len = cpos3-cpos2-1;
		MPSTdata[i].push_back(util_trim(line.substr(cpos2+1,len)));
		// Expected Peak bin Left Limit - Look for 5th comma
		cpos4 = line.find_first_of(",",cpos3+1);
		len = cpos4-cpos3-1;
		MPSTdata[i].push_back(util_trim(line.substr(cpos3+1,len)));
		// Expected Peak bin Right Limit - Look for 6th comma
		cpos5 = line.find_first_of(",",cpos4+1);
		len = cpos5-cpos4-1;
		MPSTdata[i].push_back(util_trim(line.substr(cpos4+1,len)));
		// Expected Peak Min Peak Width - Look for 7th comma
		cpos6 = line.find_first_of(",",cpos5+1);
		len = cpos6-cpos5-1;
		MPSTdata[i].push_back(util_trim(line.substr(cpos5+1,len)));

	}

	return 1;	

}

//
// ------------------------------ writeMPSTdata ------------------------------------
//
int DFMS_MPST_Class::writeMPSTdata(fstream &os, int rows, int cols) {

	string sFunctionName = "DFMS_MPST_Class::writeMPSTdata";

	char line[80];
	string sOut;

	for (int i=0; i<rows; i++) {
		sprintf(line,"%17s,%14s,%3s,%8s,%8s,%8s,%5s,%8s",	
			(MPSTdata[i][0]).c_str(), (MPSTdata[i][1]).c_str(), (MPSTdata[i][2]).c_str(), 
			(MPSTdata[i][3]).c_str(), (MPSTdata[i][4]).c_str(), (MPSTdata[i][5]).c_str(), 
			(MPSTdata[i][6]).c_str(), " ");
          //cout << line << endl;
		sOut = string(line);
		os << sOut << dfmsEOL;
	}	

	return 1;	
}

//
// ------------------------------ getMPSTdata ------------------------------------
//
int DFMS_MPST_Class::getMPSTdata() {

	string sFunctionName = "DFMS_MPST_Class::getMPSTdata";
	int nSuccess = 0;
	fstream is;

	// Verify if file is an MPST valid file
	nSuccess = isValid();
	if (!nSuccess) {
		sErrorMessage = MPSTfile+" is not a Valid MPST file\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}	

	// Open MPST file for input
	openFile(MPSTfile,is,1);

    // Read the file PDS header
    nSuccess = readPDSheader(is);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the PDS header for file: ";
		sErrorMessage += MPSTfile+"\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Read the MPST file object description
	nSuccess = readObj(is, MPSTobj, MPSTrows, MPSTcols);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the MPST Object Descp. for file: ";
		sErrorMessage += MPSTfile+"\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Read the MPST file Data
	nSuccess = readMPSTdata(is);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the MPST Data Table file: ";
		sErrorMessage += MPSTfile+"\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	} else {
		// Set the peak verification information
		L3Info->numCalRows = MPSTrows;

		// Allocate proper amount of memory to hold verification
		// peak information from MPST file
		L3Info->setvPeaksSize(MPSTrows);
		// initialize all vPeaks values
		//for (int i=0; i<MPSTrows; i++) {
		//	L3Info->vPeaks[i].init();
		//}
		for (int i=0; i<MPSTrows; i++) {
			//cout << "MPSTdata[" << i << "][0] = " << MPSTdata[i][0] << endl;
			L3Info->vPeaks[i].peakNameA = MPSTdata[i][0];
			L3Info->vPeaks[i].peakNameB = MPSTdata[i][0];
			//cout << "peakName = " << L3Info->vPeaks[i].peakName << endl;
			if (L3Info->isGCU) {
				L3Info->vPeaks[i].caltype = 0;
			} else {
				L3Info->vPeaks[i].caltype = 1;
			}
			L3Info->vPeaks[i].foundA = 0;
			L3Info->vPeaks[i].peakCntrA = 0.0;
			L3Info->vPeaks[i].peakWidthA = 0.0;
			L3Info->vPeaks[i].peakHeightA = 0.0;
			L3Info->vPeaks[i].ppmDiffA = 0.0;
			// B side
			L3Info->vPeaks[i].foundB = 0;
			L3Info->vPeaks[i].peakCntrB = 0.0;
			L3Info->vPeaks[i].peakWidthB = 0.0;
			L3Info->vPeaks[i].peakHeightB = 0.0;
			L3Info->vPeaks[i].ppmDiffB = 0.0;
		}
	}

	// Close the MPST File stream
	closeFile(is);

	return 1;
}

//
// ------------------------------ putMPSTfile ------------------------------------
//
int DFMS_MPST_Class::putMPSTfile() {

	string sFunctionName = "DFMS_MPST_Class::putMPSTfile";
	
	int rows,cols;
	fstream os;
	int nSuccess;
	
	// Open the MPST File stream
	nSuccess = openFile4Out(MPSTfile,os);
	if (nSuccess == 0) return 0;

     // write the L3 file PDS header 
     nSuccess = writePDSheader(os);
     if (!nSuccess) {
		sErrorMessage = "Unable to write the MPST PDS header for file: ";
		sErrorMessage += MPSTfile+"\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
     }
     //if (verbose == 1) printMPSTdata("PDSHeader");

     // Write the MPST file object description
     nSuccess = writeObj(os, MPSTobj, 1);
     if (!nSuccess) {
    	 sErrorMessage = "Unable to write the MPST Object Descp. for file: ";
    	 sErrorMessage += MPSTfile+"\n";
    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
 		return 0;
     }
     //if (verbose == 1) printMPSTdata("MTPobj");

     // Write the MPST file Data
     rows = util_stringToInt(MPSTobj["ROWS"]);
     cols = util_stringToInt(MPSTobj["COLUMNS"])-1;

     nSuccess = writeMPSTdata(os, rows, cols);
     if (!nSuccess) {
    	 sErrorMessage = "Unable to write the MPST Data for file: ";
    	 sErrorMessage += MPSTfile+"\n";
    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
 		return 0;
     }
     //if (verbose == 1) printMPSTdata("MPSTdata");
     // Close the MPST File stream
     closeFile(os);

     return 1;

}

//
// -------------------------------- isValid --------------------------------------
//
int DFMS_MPST_Class::isValid() {

	if ( fileName.substr(0,3) == "GCU" ||
		 fileName.substr(0,3) == "SLF" ) {
		return 1;
	} else {
		return 0;
	}

}

//
// ------------------------------ printMPSTdata ------------------------------------
//
void DFMS_MPST_Class::printMPSTobj(string type) {

	// First print the PDS Header
	
	mapstr::const_iterator it;

	if (type == "PDSHeader" || type == "All") {
		for (it = PDSheader.begin(); it != PDSheader.end(); ++it) {
			printf("%-32s =     %-s%s",(it->first).c_str(),(it->second).c_str(),EOL);
		}
	}

	if (type == "MPSTobj" || type == "All") {
		// Next print out Object description
		for (it = MPSTobj.begin(); it != MPSTobj.end(); ++it) {
			printf("%32s =     %-s%s",(it->first).c_str(),(it->second).c_str(),EOL);
		}
	}

	if (type == "MPSTdata" || type == "All") {
		// Finally print out the MPST  data
 		for (int i=0; i<MPSTrows; i++) {
			printf("%13s,%14s,%3s,%8s,%8s,%8s,%8s,%s",
						(MPSTdata[i][0]).c_str(),
						(MPSTdata[i][1]).c_str(),
						(MPSTdata[i][2]).c_str(),
						(MPSTdata[i][3]).c_str(),
						(MPSTdata[i][4]).c_str(),
						(MPSTdata[i][5]).c_str(),
						(MPSTdata[i][6]).c_str(),EOL);
		}
	}	

}

//
// ---------------------- getMPSTfilesInDir -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method retrieves all the Mass Peak Search Table files in a directory
//
// inputs: The MTP Class pointer

// returns:
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
int DFMS_MPST_Class::getMPSTfilesInDir(string sMPSTpath, vector<L2FileInfo> &MPSTinfo,
		                               string type) {
	
	string sFunctionName="DFMS_MPST_Class::getMPSTfilesInDir";

  	DIR *dir;
  	struct dirent *entry;
	string fileName;
	L2FileInfo M;
	MPSTinfo.clear();


	if ((dir = opendir(sMPSTpath.c_str())) == NULL) {
		sErrorMessage="Default Source directory - "+sMPSTpath+" does not exist";
		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
  	} else {
  		while ((entry = readdir(dir)) != NULL) {
  			fileName = string(entry->d_name);
  			string firstChar = fileName.substr(0,1);
			// Skip all files shorter than 3 characters.. (like ./, ../)
			if (fileName.size() > 3 && firstChar.compare(".") != 0) {
				string suffix = fileName.substr(fileName.find_last_of(".") + 1);
				// Only use non-empty .TAB files
				if((suffix == "tab" || suffix == "TAB")) {
					M.fileName = fileName;
					if (type.compare("GCU") == 0) {
						string smode = fileName.substr(0,3);
						// eg., GCU_MPST_20130101_010203.TAB
						string dateTime = fileName.substr(9,15);
						M.dateTime = dateTime;
						if (smode.compare("GCU") == 0) {
							M.isGCU = true;
							M.isSLF = false;
							//M.printData();
							MPSTinfo.push_back(M);
						}
					} else if (type.compare("nonGCU") == 0) {
						string smode = fileName.substr(0,3);
						// eg., SLF_MPST_20130101_010203.TAB
						string dateTime = fileName.substr(9,15);
						M.dateTime = dateTime;
						if ((smode.substr(0,3).compare("SLF")) == 0) {
							M.isGCU = false;
							M.isSLF = true;
							//M.printData();
							MPSTinfo.push_back(M);
						}
					}
				} else {
					sErrorMessage="File: "+fileName+" is not a candidate file";
					writeToLog(sErrorMessage, sFunctionName, WARN);
				}
			}
  		}
  		closedir(dir);
  	}

	if (MPSTinfo.size() == 0) {
		sErrorMessage="Default source directory - " + 
                         sMPSTpath+" does not contain expected files";
		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
	}
     
     return 1;
}

/*
//
// ----------------------- findBestMPSTfile -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method finds the closest Mass Peak Search Table file to the current
// L2 file date and time
//
// inputs: 
//   sRefDateAndTime - Input date and time
//   iL2mode - L2 file mode
//   
// returns:
//   sBestMPST - best file name found
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
int DFMS_MPST_Class::findBestMPSTfile(string sMPSTpath, string sRefDateAndTime, bool isGCU,
                     vector<L2FileInfo> &MPSTinfo)
{
	
	string sFunctionName="findBestMPSTfile";

	string nBestDateAndTime = "0";
	string nDateAndTime;
	string nThisDateAndTime;
	string sDateAndTime;

	// Initially set to MPST 
	nDateAndTime = sRefDateAndTime;
	sBestMPST = "0";

	// Filter directory contents for those that match Mass Peak Search Table 
    // file info
	for (int i=0; i<MPSTinfo.size(); i=i+1) {
		sDateAndTime = MPSTinfo[i].dateTime;
		// Compare mode MPST file with input L2 modeID 
		//if (MPSTinfo[i].mode == iL2mode) {

			nThisDateAndTime = sDateAndTime;

			int comp1 = nThisDateAndTime.compare(nDateAndTime);
			int comp2 = nThisDateAndTime.compare(nBestDateAndTime);
			if (comp1 < 0 && comp2 > 0) {
				nBestDateAndTime = nThisDateAndTime;
				sBestMPST = MPSTinfo[i].fileName;
				if (verbose >= 3) {
                        cout << "MPST - sBestMPST = " << sBestMPST << endl;
                    }
			}
		//}
		
	}

	// found no matching files
	if (sBestMPST == "0") {
		sErrorMessage = "No suitable DFMS MASS PEAK TABLE file ";
		sErrorMessage += "was found in supplied path: " + sMPSTpath;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	return 1;

}
*/

//
// ----------------------- findBestMPSTfile -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method finds the closest COPS Data file to the current
// L2 file date and time
//
// inputs:
//   sRefDateAndTime - Input date and time
//
// returns:
//   sBestCOPS - best file name found
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
int DFMS_MPST_Class::findBestMPSTfile(string sMPSTpath, string sRefDateAndTime, bool isGCU,
                     vector<L2FileInfo> &MPSTinfo)
{

	string sFunctionName="findBestMPSTfile";

	string sDateAndTime;

	// Initially set to MPST
	sBestMPST = "N/A";

	map<double,string> diffMPST;

	//
	// -------------- Look for best MPST files ---------------
	//
	if (MPSTinfo.size() > 0) {
		//cout << "MPSTinfo = " << MPSTinfo.size() << endl;
		// Filter directory contents for those that match NG COPS Data
		// file info
		for (int i=0; i<MPSTinfo.size(); i=i+1) {
			sDateAndTime = MPSTinfo[i].dateTime;
			double diff = timeDiffInSec(sDateAndTime,sRefDateAndTime);
			sBestMPST = MPSTinfo[i].fileName;
			diffMPST[diff] = sBestMPST;
		}
	}

	// No matching files found
	if (sBestMPST == "N/A") {
		sErrorMessage = "No suitable DFMS MASS PEAK TABLE file ";
		sErrorMessage += "was found in supplied path: " + sMPSTpath;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	} else {
		// Use C++ map automatic key sort property.
		// First element is the shortest time diff
		sBestMPST = diffMPST.begin()->second;
		//cout << "Found Best MPST file: " << sBestMPST << endl;
	}

	return 1;

}

/*
//
// ------------------------------ writePDSheader ------------------------------------
//
int DFMS_MPST_Class::writePDSheader(fstream &os) {

     string sFunctionName = "DFMS_MPST_Class::writePDSheader";

	char cline[80];
	string sline;

	// Write the L3 PDS Header
	writeLine(os,"PDS_VERSION_ID",PDSheader["PDS_VERSION_ID"],0);
	writeLine(os,"LABEL_REVISION_NOTE",PDSheader["LABEL_REVISION_NOTE"],1);
	writeLine(os,"RECORD_TYPE",PDSheader["RECORD_TYPE"],0);
	writeLine(os,"RECORD_BYTES",PDSheader["RECORD_BYTES"],0);
	writeLine(os,"FILE_RECORDS",PDSheader["FILE_RECORDS"],0);
	writeLine(os,"LABEL_RECORDS",PDSheader["LABEL_RECORDS"],0);
	writeLine(os,"^DFMS_MASS_PEAK_SEARCH_TABLE",PDSheader["^DFMS_MASS_PEAK_SEARCH_TABLE"],0);
	writeLine(os,"DATA_SET_ID",PDSheader["DATA_SET_ID"],1);
	writeLine(os,"DATA_SET_NAME",PDSheader["DATA_SET_NAME"],1);
	writeLine(os,"PRODUCT_ID",PDSheader["PRODUCT_ID"],0);
	writeLine(os,"PRODUCT_CREATION_TIME",PDSheader["PRODUCT_CREATION_TIME"],0);
	writeLine(os,"PRODUCT_TYPE",PDSheader["PRODUCT_TYPE"],0);
	writeLine(os,"PROCESSING_LEVEL_ID",PDSheader["PROCESSING_LEVEL_ID"],1);
	writeLine(os,"MISSION_ID",PDSheader["MISSION_ID"],0);
	writeLine(os,"MISSION_NAME",PDSheader["MISSION_NAME"],1);
	writeLine(os,"INSTRUMENT_HOST_NAME",PDSheader["INSTRUMENT_HOST_NAME"],1);
	writeLine(os,"INSTRUMENT_HOST_ID",PDSheader["INSTRUMENT_HOST_ID"],0);
	writeLine(os,"INSTRUMENT_NAME",PDSheader["INSTRUMENT_NAME"],1);
	writeLine(os,"INSTRUMENT_ID",PDSheader["INSTRUMENT_ID"],0);
	writeLine(os,"INSTRUMENT_MODE_ID",PDSheader["INSTRUMENT_MODE_ID"],0);
	writeLine(os,"^INSTRUMENT_MODE_DESC",PDSheader["^INSTRUMENT_MODE_DESC"],1);
	writeLine(os,"INSTRUMENT_TYPE",PDSheader["INSTRUMENT_TYPE"],1);
	writeLine(os,"DETECTOR_ID",PDSheader["DETECTOR_ID"],0);
	writeLine(os,"DETECTOR_DESC",PDSheader["DETECTOR_DESC"],1);
	writeLine(os,"CHANNEL_ID",PDSheader["CHANNEL_ID"],0);
	writeLine(os,"PRODUCER_ID",PDSheader["PRODUCER_ID"],0);
	writeLine(os,"PRODUCER_FULL_NAME",PDSheader["PRODUCER_FULL_NAME"],1);
	writeLine(os,"PRODUCER_INSTITUTION_NAME",PDSheader["PRODUCER_INSTITUTION_NAME"],1);

	// Add an 79 character blank line to seperate PDSheader from first Object descriptor
	sprintf(cline,"%-78s"," ");
	sline = string(cline);
	os << sline << dfmsEOL;

	return 1;

}
*/
