/* 
 * File:   DFMS_ModeID_Table_Class.cc
 * Author: marinaldi
 * 
 * Created on June 8, 2013, 4:47 PM
 */

#include "DFMS_ModeID_Table_Class.hh"

//
// ------------------------------ Constructor --------------------------------
//
DFMS_ModeID_Table_Class::DFMS_ModeID_Table_Class(string p, string f) {
    
	path = p;
	fileName = f;
	modeIDfile = path+fileName;
	modeIDrows = 0;
	modeIDcols = 0;
}

//
// ------------------------------- Destructor --------------------------------
//
DFMS_ModeID_Table_Class::~DFMS_ModeID_Table_Class() {
	modeIDobj.clear();
}

//
// ----------------------------- getFileDate ---------------------------------
//
string DFMS_ModeID_Table_Class::getFileDate() {
    
	return fileName.substr(14,8);
}

//
// ---------------------------- readModeData ---------------------------------
//
int DFMS_ModeID_Table_Class::readModeData(fstream &is) {

	string sFunctionName = "DFMS_ModeID_Table_Class::readModeData";

	char input[100];
	vector<string> fields;
	string tmp;
	int mode;
	DFMSModeInfo m;

	// Now read the mode data into the mode map
 	for (int i=0; i<modeIDrows; i++) {

		// Use DFMSmodeInfo struct and mode to create a mode/Info map
		is.getline(input,100); 
		string line = string(input);
		util_splitString(line, ',', fields);
		for (int j=0; j<fields.size(); j++) {
			string strTmp = fields[j];
			fields[j] = util_trim(util_trimChar(strTmp,"\""));
		}

		tmp = util_trim(fields[0]);
		m.sMode = tmp;
		mode = util_stringToInt(util_trim(tmp).c_str());
		m.modeType = util_trim(fields[1]);
		if ((m.modeType).compare("GCU") == 0) {
			m.isGCU = true;
		} else {
			m.isGCU = false;
		}
		m.detector = util_trim(fields[2]);
		m.res = util_trim(fields[3]);
		if ((m.res).compare("N/A") != 0) {
			m.res = (m.res).substr(0,1);
		}
		if ((m.res).compare("L") == 0) {
			m.lowRes = true;
			m.hiRes = false;
		} else {
			m.lowRes = false;
		}
		if ((m.res).compare("H") == 0) {
			m.hiRes = true;
			m.lowRes = false;
		} else {
			m.hiRes = false;
		}
		m.sens = util_trim(fields[4]);
		if ((m.sens).compare("N/A") == 0) m.emission = 0.0;
		if ((m.sens).compare("LOWSENS") == 0) m.emission = 2.0e-06;
		if ((m.sens).compare("MEDSENS") == 0) m.emission = 20.0e-06;
		if ((m.sens).compare("HIGHSENS") == 0) m.emission = 200.0e-06;
		m.modeDesc = util_trim(fields[5]);
		m.zoom = util_stringToDouble(util_trim(fields[6]));

		//m.printData();
		//cout << endl;
		modeData.insert(pair<int,DFMSModeInfo>(mode,m));
        fields.clear();

	}

	return 1;	

}

// ----------------------- findLatestModeTable ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method finds the closest COPS Data file to the current
// L2 file date and time
//
// inputs:
//   sRefDateAndTime - Input date and time
//
// returns:
//   sBestCOPS - best file name found
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
string DFMS_ModeID_Table_Class::findLatestModeTable(string sModeIDTablePath)
{

	string sFunctionName="findLatestModeFile";
	DIR *dir;
  	struct dirent *entry;
  	string fileName, fileLatest;

	double timeTag=0.0, tLatest=0.0;

  	int k=0;

  	// This is if the files are all in one folder
  	string searchPath = sModeIDTablePath;

  	// open the mode id table directory and search for the latest file
  	if ((dir = opendir(searchPath.c_str())) == NULL) {
  		sErrorMessage="Default Source directory - "+searchPath+" does not exist";
  		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
	} else {
		while ((entry = readdir(dir)) != NULL) {
			fileName = string(entry->d_name);
			if (fileName.length() > 27) {
				string prefix = fileName.substr(0,18);
				string firstChar = fileName.substr(0,1);
				string suffix = fileName.substr(fileName.find_last_of(".") + 1);
				// Skip all files shorter than 3 characters.. (like ./, ../)
				if ((prefix.compare("DFMS_MODE_ID_TABLE") == 0) &&
						(suffix == "tab" || suffix == "TAB") &&
					     firstChar.compare(".") != 0) {
					k++;
					//cout << "filename = " << fileName << endl;
					string dateTime = fileName.substr(19,8);
					cout << "dateTime = " << dateTime << endl;
					timeTag = rosToSec(dateTime,2);
					// Look for latest mode ID table
					if (timeTag > tLatest) {
						tLatest = timeTag;
						fileLatest = fileName;
					}
				} else {
					sErrorMessage="File: "+fileName+" is not a candidate file";
					writeToLog(sErrorMessage, sFunctionName, WARN);
				}
	  		} else {
	  			sErrorMessage="File: "+fileName+" is not a candidate file";
	  			writeToLog(sErrorMessage, sFunctionName, WARN);
	  		}
	  	}
		// If no modeID files were found then exit badly.
		if (tLatest == 0.0) {
			sErrorMessage="No candidate modeID files found - Exit";
			writeToLog(sErrorMessage, sFunctionName, FATALERROR);
			toLog.close();
			exit(-1);
		}
		closedir(dir);
	}

	sInfoMessage="Latest ModeID Table found is: " + fileLatest;
	writeToLog(sInfoMessage, sFunctionName, INFO);
  	cout << sInfoMessage <<endl;

	return fileLatest;

}

//
// ----------------------------- getModeData ---------------------------------
//
int DFMS_ModeID_Table_Class::getModeData() {

     string sFunctionName = "DFMS_ModeID_Table_Class::getModeData";
	
	fstream is;
	int nSuccess;
	
	// Open the mode ID File stream
	openFile(modeIDfile,is,1);

        // Read the mode ID file PDS header 
     nSuccess = readPDSheader(is);
     if (!nSuccess) {
    	 sErrorMessage = "Unable to read the Mode ID PDS header for file: ";
    	 sErrorMessage += modeIDfile+" - Skipping file processing";
    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
    	 return 0;
     }
     //if (verbose == 1) printModeIDfile("PDSHeader");

     // Read the Mode ID file Data object description
     nSuccess = readObj(is, modeIDobj, modeIDrows, modeIDcols);
     if (!nSuccess) {
    	 sErrorMessage = "Unable to read the Data Object Descp. for file: ";
    	 sErrorMessage += modeIDfile+" - Skipping file processing";
    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
    	 return 0;
     }
     //if (verbose == 1) printModeIDfile("modeIDobj");

     // Read the Mode ID file Data
     nSuccess = readModeData(is);
     if (!nSuccess) {
    	 sErrorMessage = "Unable to read the Mode ID Data for file: ";
    	 sErrorMessage += modeIDfile+" - Skipping file processing";
    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
    	 return 0;
     }
     //if (verbose == 1) printModeIDfile("modeIDdata");

     // Close the Mode ID File stream
     closeFile(is);
 
     return 1;
}

//
// -------------------------- printModeFile ----------------------------------
//
void DFMS_ModeID_Table_Class::printModeFile(string type) {

	// First print the PDS Header
	
	map<string,string>::const_iterator it;

	if (type == "PDSHeader" || type == "All") {
		for (it = PDSheader.begin(); it != PDSheader.end(); ++it) {
			printf("%-32s =     %-s%s",(it->first).c_str(),(it->second).c_str(),EOL);
		}
		cout << endl;
	}

	// Next print out modeIDobj Object description
	if (type == "modeIDobj" || type == "All") {
		for (it = modeIDobj.begin(); it != modeIDobj.end(); ++it) {
			printf("%32s =     %-s%s",(it->first).c_str(),(it->second).c_str(),EOL);
		}
		cout << endl;
	}

     // Finally print out the data
	if (type == "modeIDdata" || type == "All") {

		for (int i=0; i<modeIDrows; i++) {
			printf("%5.5s,%5.5s,%5.5s,%8.8s,%8.8s,%23.23s,%5.5f%s",modeData[i].sMode.c_str(),
                       modeData[i].modeType.c_str(), modeData[i].detector.c_str(),
                       modeData[i].res.c_str(), modeData[i].sens.c_str(),
                       modeData[i].modeDesc.c_str(), modeData[i].zoom, EOL);
		}	

	}

}
