
#include "DFMS_PDS_L2_file_Class.hh"

//
// ---------------------------- Constructor ------------------------------------
//
DFMS_PDS_L2_file_Class::DFMS_PDS_L2_file_Class(string p, string f) {
     
	path = p;
	fileName = f;
	L2File = path+fileName;
	fileDate = getFileDate();
	HKrows = 0;
	dataRows=0;
	dataCols=0;

}

//
// ---------------------------- Destructor -------------------------------------
//
DFMS_PDS_L2_file_Class::~DFMS_PDS_L2_file_Class() {
	HKobj.clear();
	DataObj.clear(); 
	HKdata.clear();
	L2Data.clear();
	DataObj.clear();
}

//
// ----------------------------- getFileDate -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Returns the L2 file data string
//
// inputs: None
//
// returns:
//   the date string
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
string DFMS_PDS_L2_file_Class::getFileDate() {

	return fileName.substr(3,18);
}

//
// ----------------------------- int getModeID -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Returns the L2 file DFMS mode number
//
// inputs: None
//
// returns:
//   the integer mode
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
int DFMS_PDS_L2_file_Class::getModeID() {

	return util_stringToInt(fileName.substr(24,4));
}

//
// ----------------------------- string getModeID -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Returns the L2 DFMS mode string
//
// inputs: None
//
// returns:
//   the mode string
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
string DFMS_PDS_L2_file_Class::getStrModeID() {

	return fileName.substr(23,5);
}

//
// ------------------------------ readHKdata ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Reads the HouseKeeping data from the L2 file
//
// inputs:
//  is  -  Reference to the input stream
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
int DFMS_PDS_L2_file_Class::readHKdata(fstream &is) {

	string sFunctionName = "DFMS_PDS_L2_file_Class::readHKdata";

	char input[100];
	int cpos0,cpos1,cpos2;   // positions of commas within string/substring
	int len;

	// Now read the HK data into the HKdata string array
 	for (int i=0; i<HKrows; i++) {

		// Use vector objects dynamic fill to create one row
		HKdata.push_back(sslice());

		is.getline(input,100); 
		string line = string(input);
		if (line.length() > 85) {
			sErrorMessage = "Possible L2 HouseKeeping line overrun in";
			sErrorMessage += "L2 file "+fileName;
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			is.close();
			skipReas["L2HKLineOverun"]++;
			return 0;
		}
		// Look for 1st comma
		cpos0 = line.find_first_of(",",0);
		len = cpos0-2;
		HKdata[i].push_back(util_trim(line.substr(1,len)));

		// Look for second comma
		cpos1 = line.find_first_of(",",cpos0+1);
		len = cpos1-cpos0-3;
		HKdata[i].push_back(util_trim(line.substr(cpos0+2,len)));

 		// Look for third and last comma
		cpos2 = line.find_first_of(",",cpos1+1);
		len = cpos2-cpos1-3;
		HKdata[i].push_back(util_trim(line.substr(cpos1+2,len)));

		len = 5;
		HKdata[i].push_back(util_trim(line.substr(cpos2+2,len)));

	}

	return 1;	

}

//
// ------------------------------ readL2data ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Reads the Count data from the L2 file
//
// inputs:
//  is  -  Reference to the input stream
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
int DFMS_PDS_L2_file_Class::readL2data(fstream &is) {

	string sFunctionName = "DFMS_PDS_L2_file_Class::readL2data";

	char input[100];

	// Now read the HK data into the HKdata string array
 	for (int i=0; i<dataRows; i++) {
		L2Data.push_back(islice());
		for (int j=0; j<dataCols-1; j++) {
			is.getline(input,100,','); 
			L2Data[i].push_back(util_stringToInt(input));
		}
	}

	return 1;	
}

//
// -----------------------------------------  getL2file ----------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Performs all the steps need to open/read/close and L2 file and store the
// various data into arrays
//
// inputs:
//  L2Info  -  Reference to L2 file info structure
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
int DFMS_PDS_L2_file_Class::getL2file(L2FileInfo &L2Info) {

	string sFunctionName = "DFMS_PDS_L2_file_Class::getL2file";
	
	fstream is;
	int nSuccess;
	
	// Open the L2 File stream
	nSuccess = openFile(L2File,is,1);
	if (!nSuccess) {
		sErrorMessage = "Unable to open L2 PDS file: ";
		sErrorMessage += L2File+" - Skipping file processing";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		if (is.is_open()) is.close();
		return 0;
	}

    // Read the L2 file PDS header
	nSuccess = readPDSheader(is);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the PDS header for file: ";
		sErrorMessage += L2File+" - Skipping file processing";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
  	//cout << "NOTE:" << endl;
  	//cout << PDSheader["NOTE"] << endl;
  	//exit(0);
	//if (verbose == 1) printL2File("PDSHeader");

	// Read the L2 file HK object description
	nSuccess = readObj(is, HKobj, HKrows, HKcols);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the HK Object Descp. for file: ";
		sErrorMessage += L2File+" - Skipping file processing";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printL2File("L2HKobj");

	// Read the L2 file Data object description
	//int dataRows = NUMPIX;
	nSuccess = readObj(is, DataObj, dataRows, dataCols);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the Data Object Descp. for file: ";
		sErrorMessage += L2File+" - Skipping file processing";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printL2File("L2dataObj");

	// Read the L2 file HK Data
	nSuccess = readHKdata(is);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the HK Data for file: ";
		sErrorMessage += L2File+" - Skipping file processing";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	//if (verbose == 1) printL2File("L2HKdata");
	//printL2File("L2HKdata");

	// Read the L2 file L2 Data
	nSuccess = readL2data(is);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the L2 Data for file: ";
		sErrorMessage += L2File+" - Skipping file processing";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printL2File("L2data");

	// Close the L2 File stream
	closeFile(is);
     
	if (verbose >= 3) L2Info.printData();

	return 1;

}

//
// ---------------------------- getHKdataVal ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Reads a specific HouseKeeping data value from the L2 file
//
// inputs:
//  s  -  map string to read
//  j  -  map array value to return
//
// returns:
//   the ap value pertaining to string s
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
string DFMS_PDS_L2_file_Class::getHKdataVal(string s, int j) {
    
    int rows = HKdata.size();
    for(int i=0; i<rows; ++i) {
        if ((HKdata[i][0]).compare(s) == 0) {
            return HKdata[i][j];
            break;
        }
    }
    
    // If you got here it did not find the value
    return "0";
}

//
// ---------------------------- printL2File ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Prints the L2 file values to stdout
//
// inputs:
//  type  -  the PDS part of the file to print
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
void DFMS_PDS_L2_file_Class::printL2File(string type) {

	// First print the PDS Header
	
	map<string,string>::const_iterator it;

	if (type == "PDSHeader" || type == "All") {
		for (it = PDSheader.begin(); it != PDSheader.end(); ++it) {
			printf("%-32s =     %-s%s",(it->first).c_str(),(it->second).c_str(),EOL);
		}
		cout << endl;
	}

	// Next print out HK Object description
	if (type == "L2HKobj" || type == "All") {
		for (it = HKobj.begin(); it != HKobj.end(); ++it) {
			printf("%32s =     %-s%s",(it->first).c_str(),(it->second).c_str(),EOL);
		}
		cout << endl;
	}

	// Next print out Data Object description
	if (type == "L2dataObj" || type == "All") {
		for (it = DataObj.begin(); it != DataObj.end(); ++it) {
			printf("%32s =     %-s%s",(it->first).c_str(),(it->second).c_str(),EOL);
		}
		cout << "END" << endl;
	}

	// Next print out the HK data
	//cout << "HKdata[0][0] = " << HKdata[0][0] << endl;
	if (type == "L2HKdata" || type == "All") {
	 	for (int i=0; i<HKrows; i++) {
			printf("\"%-32s\",\"%-5s\",\"%-15s\",\"%-5s\"%s",
				(HKdata[i][0]).c_str(),
				(HKdata[i][1]).c_str(),
				(HKdata[i][2]).c_str(),
				(HKdata[i][3]).c_str(),EOL);
		}
	}

	// Finally print out the data
	if (type == "L2data" || type == "All") {
		for (int i=0; i<NUMPIX; i++) {
			printf("%3d,%12d,%12d,%s",L2Data[i][0], L2Data[i][1], L2Data[i][2], EOL);
		}	
	}

}

//
// --------------------------------------- DumpL2toFile ----------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Dumps L2 raw data values from the L2 file
//
// inputs:
//  path  -  Output path for dump
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
int DFMS_PDS_L2_file_Class::DumpL2toFile(string path) {

	string file=path + fileName.substr(0,fileName.length()-4)  + "_plot.dat";
	fstream os;
	os.open(file.c_str(), fstream::out | fstream::binary);
	os << "Pixel    A-counts    B-counts" << dfmsEOL;
	char tmp[80];
	for (int i=0; i<512; ++i) {
		sprintf(tmp,"%5d    %8d    %8d",L2Data[i][0],
        		  L2Data[i][1],L2Data[i][2]);
		os << string(tmp) << dfmsEOL;
	}

	os.close();

	return 1;
}
