

#include "DFMS_PDS_file_Class.hh"

//
// ---------------------------- Constructor ------------------------------------
//
DFMS_PDS_file_Class::DFMS_PDS_file_Class() {

	lastPDSHeaderLine = 0;
	numObjs = 0;
	lblRecs = 0;

}

//
// ----------------------------- Destructor ------------------------------------
//
DFMS_PDS_file_Class::~DFMS_PDS_file_Class() {
    
	// Release dynamically allocated memory
	PDSheader.clear();
	pdsFileName.clear();
}

//
// ---------------------------- Open PDS file ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Opens a PDS file
//
// inputs:
//	file - full path name to file
//  str  - reference to input or output stream
//  in   - if 1 then open to read
//         if 0 then open to write
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
//
int DFMS_PDS_file_Class::openFile(string file, fstream &str, int in) {

	string sFunctionName = "DFMS_PDS_file_Class::openFile";

	pdsFileName = file;

	// Open the PDS file input stream for reading
	if (in) {
		//cout << "file = " << file << endl;
		str.open(file.c_str(), fstream::in);
		if(!str.is_open()) {
			sErrorMessage = "Error opening for reading PDS File: "+file;
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			return 0;
		}
	} else {
		str.open(file.c_str(), fstream::out | fstream::binary);
		if(!str.is_open()) {
			sErrorMessage = "Error opening for writing PDS File: "+file;
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			return 0;
		}
	}

	return 1;
}

//
// ---------------------------- Close PDS file ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Closes a PDS file
//
// inputs:
//  is - reference to input or output stream
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
void DFMS_PDS_file_Class::closeFile(fstream &is) {

	// close the L2 file input stream
	is.close();
}

//
// ------------------------------ readPDSheader ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Read a PDS header from PDS file
//
// inputs:
//  is - reference to input or output stream
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
int DFMS_PDS_file_Class::readPDSheader(fstream &is) {

	string sFunctionName = "DFMS_PDS_file_Class::readPDSheader";

	char input[100];
	string line;
	string key, value;
	int lineNum = 0;
     
	// initialize tokens vector to no tokens
	line.clear();

	// Find the number of data descriptor objects and the label record value
	lastPDSHeaderLine = findLastPDSLine();
	if (lastPDSHeaderLine <= 0) {
		sErrorMessage = "lastPDSHeaderLine calculation returned <= 0.";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Loop through all the PDS Header and create the key/value map
	for (int i=1; i<lastPDSHeaderLine+1; i++) {
 		is.getline(input,100);
 		if (is.eof()) break;
		lineNum++;
		line.assign(input);
		//cout << "line " << i << ": " << line << endl;
 		if (i == lastPDSHeaderLine) break;
		int epos = line.find_first_of("=");
		// Add additional check for location of "=".
		// This makes sure that the equals we're interested in
		// is between position 25 and 40
		if (epos != string::npos && epos < 40 && epos > 25) {
			// Found an = sign
			key = util_trim(line.substr(0,epos-1));
			value = util_trim(line.substr(epos+1,string::npos));
			PDSheader[key] = value;
			//cout << "key=" << key << "  value = " << value << endl;
		// If there is no "=" or if it is outside the position bounds
		// of > 25 and < 40 then assume a continuation line
		} else if ((epos > 40 && util_isalnum(line)) ||
				   (epos == string::npos && util_isalnum(line))) {
			// Found a continuation line
			value = "\n"+line;
			PDSheader[key] += value;
			//cout << "key=" << key << "  value = " << value << endl;
		} else if (!util_isalnum(line) ) {
			//Found a valid blank line that will be added to value
			value = "\n"+line;
			PDSheader[key] += value;
		}
    	// clear string for next time
		line.clear();
	}

	return 1;

}

//
// --------------------------------- readObj --------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Read an Object description from PDS file
//
// inputs:
//  is - reference to input stream
//  descObj - string,value map pairs describing the object
//  rows - Number of rows in the object
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// Modified: added return of cols, May 2016
// =============================================================================
int DFMS_PDS_file_Class::readObj(fstream &is, mapstr &descObj, int &rows, int &cols) {

	string sFunctionName = "DFMS_PDS_file_Class::readObj";

	char input[100];
	string line, word;
	string key, value;
	int lineNum = 0;

 	while (true) {
 		is.getline(input,100);
 		if (is.eof()) break;
		line.assign(input);
		//cout << "line = " << line << endl;
		// Remove line breaks
		line.erase( std::remove(line.begin(), line.end(), '\r'), line.end() );
		int lineLen = util_trim(line).length();
		int epos = line.find_first_of("=");
		lineNum++;
		if (epos != string::npos) {
			// Found an = sign
			key = util_trim(line.substr(0,epos-1));
			value = util_trim(line.substr(epos+1,string::npos));
			descObj[key] = value;
		} else if (util_trim(line) == "END" || lineLen == 0) {
			// reached end of the Desc. object
			break;              
        } else {
        	if (lineNum == 1 && key != "OBJECT") {
        		sErrorMessage = "Line 1 of Desc Object dose not conform to standards";
				writeToLog(sErrorMessage, sFunctionName, ERROR);
				return 0;
			} else {
				sErrorMessage = "Unknown condition while reading desc Object";
				writeToLog(sErrorMessage, sFunctionName, ERROR);
                return 0;
			}
		}
         
	}

	rows = util_stringToInt(descObj["ROWS"]);
	cols = util_stringToInt(descObj["COLUMNS"]);

	return 1;		
}

//
// --------------------------------- writeObj --------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Write an Object description to PDS file
//
// inputs:
//  os - reference output stream
//  descObj - string,value map pairs describing the object
//  last - 1 if this is the last object to write, 0 otherwise
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
int DFMS_PDS_file_Class::writeObj(fstream &os, mapstr &descObj, int last) {

	string sFunctionName = "DFMS_PDS_file_Class::writeObj";

	char line[80];
	string value;

	sprintf(line,"%-33s=     %-39s","OBJECT",(descObj["OBJECT"]).c_str());
	os << string(line) << dfmsEOL;
	sprintf(line,"   %-30s=     %-39s","NAME",(descObj["NAME"]).c_str());
	os << string(line) << dfmsEOL;
	sprintf(line,"   %-30s=     %-39s","INTERCHANGE_FORMAT",(descObj["INTERCHANGE_FORMAT"]).c_str());
	os << string(line) << dfmsEOL;
	sprintf(line,"   %-30s=     %-39s","ROWS",(descObj["ROWS"]).c_str());
	os << string(line) << dfmsEOL;
	sprintf(line,"   %-30s=     %-39s","COLUMNS",(descObj["COLUMNS"]).c_str());
	os << string(line) << dfmsEOL;
	sprintf(line,"   %-30s=     %-39s","ROW_BYTES",(descObj["ROW_BYTES"]).c_str());
	os << string(line) << dfmsEOL;
	sprintf(line,"   %-30s=     %-39s","^STRUCTURE",(descObj["^STRUCTURE"]).c_str());
	os << string(line) << dfmsEOL;
	sprintf(line,"%-33s=     %-39s","END_OBJECT",(descObj["END_OBJECT"]).c_str());
	os << string(line) << dfmsEOL;

	if (last) {
		value = "END";
	} else {
		value = "";
	}

	sprintf(line,"%-78s",value.c_str());
	os << line << dfmsEOL;

	return 1;		
}

//
// --------------------------- findLastPDSLine --------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Find the position of the last PDS header line.  Return the line number
//
// inputs:
//  file - PDS file to read
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, Aug 2015
// =============================================================================
int DFMS_PDS_file_Class::findLastPDSLine() {

	string sFunctionName = "DFMS_PDS_file_Class::findLastPDSLine";

	fstream in;
	char input[100];
	string line="";
	int numLines=0;
	vector<string> tokens;
	// initialize tokens vector to no tokens
	line.clear();

	//cout << "pdsFileName: = " << pdsFileName << endl;
	numObjs = 0;
	lblRecs = 0;

	// Open the file for reading
	in.open(pdsFileName.c_str(), fstream::in);
	if(!in.is_open()) {
		sErrorMessage = "Error opening for reading PDS File: "+pdsFileName;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	} else {

		// Read the each line and look for OBJECT/END_OBJECT keyword pair
		while (true) {
			in.getline(input,100);
			numLines++;
			if (in.eof()) break;
			line.assign(input);
			//cout << "line " << numLines << ": "<< line << endl;
			string obj = line.substr(0,6);
			// If I found an OBJECT then check if 7 lines later
			// I find END_OBJECT.  If so, then I found a data object
			// descriptor.
			if (obj.compare("OBJECT") == 0) {
				//cout << "found an object keyword: " << obj << endl;
				for (int i=1; i<8; i++) {
					in.getline(input,100);
					numLines++;
				}
				line.assign(input);
				string obj = line.substr(0,10);
				if (obj.compare("END_OBJECT") == 0) {
					//cout << "found an end_object keyword at line " << numLines << ": " << obj << endl;
					numObjs++;
				}
			}
			// Look for LABEL_RECORDS value
			string lbl = line.substr(0,13);
			if (lbl.compare("LABEL_RECORDS") == 0) {
				util_splitString(line, '=', tokens);
				lblRecs = util_stringToInt(util_trim(util_trim(tokens[1])));
				tokens.clear();
			}
		}
	}

	in.close();

	return (lblRecs - 9*numObjs);

}
