
#include "DFMS_PeakFit_Class.hh"

//  This Class is the C++ version of the fitting related functions in the RTOF
//  software of the file BAM_curvefit.c.  It includes routines for setting up
//  and executing a non-linear curve fit of a 2-dimensional data set to a function
//  defined in the method 'fitFunc", which shall be modified y the user for the specific
//  purpose of the software this has been bundled with.  The Marquardt algorithm
//  is used as described in P.R. Bevington, "Data Reduction and Error Analysis 
//  for the Physical Sciences," McGraw-Hill, 1969.

//
// ---------------------------- Constructor ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This is the PeakFit constructor. It sets up the Class data inputs
// needed to fit either a 3 or 4 term Gaussian or Gaussian plus linear term to 
// the input L2 data.
//
// inputs:
//   iRow - indicating which side the Y should represent
//   yt - the counts
//   nP - number of parameters used in fir (3 or 4 at present)
//   MPSTd - MPST data array
//   off - Mean Offset
//   t - threshold
//   rows - MPST Data rows
//   il - pixel start value
//   ir - pixel ebd value

// returns:
//   none
// =============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
// =============================================================================
//
DFMS_PeakFit_Class::DFMS_PeakFit_Class(int iRow, double* yt, int nP,
							str2D &MPSTd, double off, double t,
							int rows, int il, int ir)
{

	string sFunctionName = "DFMS_PeakFit_Class::Constructor";

	yAmaxId = 0;
	yBmaxId = 0;

	double ymax = -1.0e99;

	fitSucceeded = true;

    meanOffset = off;
    threshold = t;


    // Set up the peak area arrays.  These are used in the
    // Histogram calls find the peak characteristics
    iXl = il;
    iXr = ir;
    nPeakRange = iXr-iXl+1;

	XpartIn = new double[nPeakRange];
	YpartIn = new double[nPeakRange];
	int k=0;
	for(int i=iXl; i<iXr+1; ++i) {
		XpartIn[k] = (double)i;
		YpartIn[k] = yt[i];
		//cout << "YpartIn[" << k << "] = " << YpartIn[k] << endl;
		if (iRow == 1) {
			if (YpartIn[k] > ymax) {
				ymax = YpartIn[k];
				yAmaxId = i;
			}
		} else {
			if (YpartIn[k] > ymax) {
				ymax = YpartIn[k];
				yBmaxId = i;
			}
		}
		k++;
	}

	// Set up the full array for regular curve fit processing
	Xin = new double[NUMPIX];
	Yin = new double[NUMPIX];
	yAin = new double[NUMPIX];
	yBin = new double[NUMPIX];
	yFitA = new double[NUMPIX];
	yFitB = new double[NUMPIX];
	for(int i=0; i<NUMPIX; ++i) {
		Xin[i] = (double)(i+1);
		Yin[i] = yt[i];
		if (iRow == 1) {
			yAin[i] = Yin[i];
			yFitA[i] = 0.0;
		} else {
			yBin[i] = Yin[i];
			yFitB[i] = 0.0;
		}
	}


	nParams = nP;
	MPSTrows = rows;
    
	// Allocate memory for fit coefficients
	coeff = new double[nParams];
	// Allocate memory for error on fit coefficients
	errCoeff = new double[nParams];
	// Allocate memory to weight array
	weight = new double[NUMPIX];
	// Allocate memory to fit array
	yFit = new double[NUMPIX];

	// Create local MPST data array
	for (int i=0; i<MPSTrows; ++i) {
		MPSTdata.push_back(sslice());
		for (int j=0; j<7; ++j) {
			MPSTdata[i].push_back(MPSTd[i][j]);
		}
	}
     
}

//
// ------------------------------ Destructor -----------------------------------
//
DFMS_PeakFit_Class::~DFMS_PeakFit_Class() {
    
    // Release dynamically allocated memory
    delete[] XpartIn;    XpartIn=0;
    delete[] YpartIn;   YpartIn=0;
    delete[] yAin;	yAin=0;
    delete[] yBin;	yBin=0;
    delete[] Xin;    Xin=0;
    delete[] Yin;   Yin=0;
    delete[] coeff;  coeff=0;
    delete[] errCoeff;  errCoeff=0;
    delete[] weight; weight=0;
    delete[] yFit;   yFit=0;
    delete[] yFitA;   yFitA=0;
    delete[] yFitB;   yFitB=0;
    MPSTdata.clear();
    
}
//
// ------------------------------- initPeakFit ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method will initialize the fit coefficients as follows,
//   coeff[0] term:  Use the max data value as the peak
//   coeff[1] term:  Use the location of the peak+1 as the center pixel
//   coeff[2] term:  Default guess, 10.0 pixels
//
// inputs:  Uses class PeakFit input class field data YAin or YBin.
//		iRow - DFMS side A/B
//
// returns:
//   none
// =============================================================================
// History: Written by Mike Rinaldi, March 2013.
// =============================================================================
//
int DFMS_PeakFit_Class::initPeakFit(int iRow, int pkLoc, double pkHght,
													double pkWd, bool abvThresh) {

	string sFunctionName = "DFMS_PeakFit_Class::initPeakFit";
     
	DFMS_Histogram_Class *hPeak;
	// Use histogram class to find initial peak
	// fit coefficients and background

	// 2.3.1 -  Use part of array around expected peak location to initialize peak coeff
	hPeak = new DFMS_Histogram_Class(YpartIn, nPeakRange, HISTONUMBINS, string("peak"));

	// 2.3.2 - Make the peak region histogram (used to find peak)
	hPeak->createHisto(iRow);
     
    // 2.3.3 - Set up the initial coefficients guesses
	//coeff[0] = pow(10.0,hPeak->hMax);  	// Peak height
	coeff[0] = hPeak->hMax;  				// Peak height
	coeff[1] = (double)hPeak->iMax+iXl+1;   // Peak (+Xl because array is offset)
	coeff[1] = (double)(pkLoc+1);
	coeff[2] = 9.0;                    		// FWHM (use 5 if low peak 9 if tall)
	//cout << "Row A - Initial Peak pixel loc    coeff[1] = " << coeff[1] << endl;

	if (verbose == 10) {
		if (iRow == 1) {
			cout << "Row A - Initial Peak Ions/s       coeff[0] = " << coeff[0] << endl;
			cout << "Row A - Initial Peak pixel loc    coeff[1] = " << coeff[1] << endl;
			cout << "Row A - Initial Peak Width        coeff[2] = " << coeff[2] << endl;
		} else {
			cout << "Row B - Initial Peak Ions/s       coeff[0] = " << coeff[0] << endl;
			cout << "Row B - Initial Peak pixel loc    coeff[1] = " << coeff[1] << endl;
			cout << "Row B - Initial Peak Width        coeff[2] = " << coeff[2] << endl;
		}
	}

	// 2.3.4 - Check if peak is Above threshold
	if (!abvThresh) {
		sInfoMessage = "Alert: No reasonable Peak Fond for "+sRow[iRow-1];
		writeToLog(sErrorMessage, sFunctionName, WARN);
		L3Info->Pix0Calc = false;
	    delete hPeak; hPeak=0;
	    return -1;
	}

	// 2.3.4 - Check if Peak in within bounds
	if (coeff[1] < PKLOWPT || coeff[1] > PKHIGHPT) {
		if (!CONVERTOUTBPEAK || (coeff[1] < 20 || coeff[1] > 492)) {
            if (iRow == 1) sideAexceptionType["peakOutOfBounds"]++;
            if (iRow == 2) sideBexceptionType["peakOutOfBounds"]++;
			sErrorMessage = "Peak is outside of bounds";
			writeToLog(sErrorMessage, sFunctionName, WARN);
			delete hPeak; hPeak=0;
			if (QLINFOFILE) {
				QuickLookLog << " - Skipped - Peak Location: " <<  coeff[1] << " is out of Bounds" << dfmsEOL;
				qlCount++;
				if (qlCount % QLPPAGE == 0) qlHeader();
			}
			cout << " - Skipped - Peak Location: " <<  coeff[1] << " is out of Bounds" << endl;
			return 0;
		} else {
			sInfoMessage = "Alert: Peak is outside of bounds: "+util_doubleToString(coeff[1]);
			writeToLog(sErrorMessage, sFunctionName, WARN);
			L3Info->Pix0Calc = false;
			if (iRow == 1) sideAexceptionType["peakOutOfBounds"]++;
			if (iRow == 2) sideBexceptionType["peakOutOfBounds"]++;
			fitSucceeded = checkHistoSmoothness(hPeak, iRow);
		}
	}

	// 2.3.5 - Check if peak height is sufficient to warrant use for pix0 calibration
	double pHlimit;
	if (meanOffset + PKHEIGHTLIMIT >= threshold) {
		pHlimit = threshold;                 // This is based on NUMSDEV*stdDevMeanOffset
	} else {
		pHlimit = meanOffset + PKHEIGHTLIMIT;
	}
	if (pkHght < pHlimit) {
		if (!CONVERTLOWAMP) {
			sErrorMessage = "Peak is less than mean Offset + " +
					util_doubleToString(NUMSDEV)+"*meanOffsetStdDev";
			skipReas["peakHeightLimit"]++;
			writeToLog(sErrorMessage, sFunctionName, WARN);
			delete hPeak; hPeak=0;
			if (QLINFOFILE) {
				QuickLookLog << " - Skipped - Peak is less than mean Offset + " +
						util_doubleToString(NUMSDEV)+"*meanOffsetStdDev" << dfmsEOL;
				qlCount++;
				if (qlCount % QLPPAGE == 0) qlHeader();
			}
			cout << " - Skipped - Peak is less than mean Offset + " +
					util_doubleToString(NUMSDEV)+"*meanOffsetStdDev" << endl;
			return 0;
		} else {
			sInfoMessage = "Alert: Peak is less than mean Offset + " +
					util_doubleToString(NUMSDEV)+"*meanOffsetStdDev";
			writeToLog(sErrorMessage, sFunctionName, WARN);
			L3Info->Pix0Calc = false;
		    fitSucceeded = false;
		    if (iRow == 1) sideAexceptionType["peakHeightLimit"]++;
		    if (iRow == 2) sideBexceptionType["peakHeightLimit"]++;
		}
	}
        
	// TODO - fix numbering
	// 2.3.6 - Release memory
    delete hPeak; hPeak=0;

    return 1;
     
}

//
// -------------------------------- fitFunc ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method is the user supplied fit function. It is either a 3 term or 4 term
// Gaussian or Gaussian plus linear term for background.
//
// inputs:
//   x - independent variable
//   c - fit parameters (coefficient) 
//
// returns:
//   The corresponding Gaussian function value for this x
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
double DFMS_PeakFit_Class::fitFunc(double x, double* c) {

	string sFunctionName = "DFMS_PeakFit_Class::fitFunc";

	double y = 0.0;

	if (nParams == 3) {
		// Gaussian function, F = A[0]*exp((-((X-A[1])/A[2])^2.)/2.)
		y = coeff[0]*exp(-1.0 * pow(((x - coeff[1])/coeff[2]),2.0) / 2.0); //+ meanOffset;
	} else if (nParams == 4) {
		// Gaussian function, F = A[0]*exp((-((X-A[1])/A[2])^2.)/2.) + A[3]
		y = coeff[0]*exp(-1.0 * pow(((x - coeff[1])/coeff[2]),2.0) / 2.0) +
				meanOffset;
	} else {
		sErrorMessage = "Only Gaussian for for N=3 or 4 is supported by ";
		sErrorMessage += "the current fit function\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
	}

	return y;
}

//
// ------------------------------ calcDeriv ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine calculates the numerical derivatives held in the Derivatives matrix
// used by the curveFit algorithm
//
// inputs:
//   yFit - uses class variable yFit 
//
// modifies:
//   returns with new calculated values according to the difference between new
//   and previously fitted Y-values (modifying the function parameters to obtain 
//   the new values)
// returns;
//   None
// =============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
// =============================================================================
//
void DFMS_PeakFit_Class::calcDeriv(double** derivs) {

	string sFunctionName = "DFMS_PeakFit_Class::calcDeriv";

	double nParamTmp;
	double nDelta;
	double nYfit;
	for (int j=0; j<nParams; ++j) {
		nParamTmp = coeff[j];
		nDelta = fabs(coeff[j] / 100000);
		coeff[j] = nParamTmp + nDelta;
		for (int i=0; i<NUMPIX; ++i) {
			nYfit = fitFunc(Xin[i], coeff);
			derivs[i][j] = (nYfit - yFit[i]) / nDelta;
			//cout << "derivs[" << i << "][" << j << "] = " << derivs[i][j] << endl;
		}
		coeff[j] = nParamTmp;
	}
     
}

//
// ---------------------------- calcWeights ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine calculates the initial weights of the data using 1/sigma^2.  
// where sigma^2 is defined as SUMj (Yi - avgYi)^2.  here we're summing over
// replicated data.  The data is not really replicated but I'll assume that 
// side A and B data are replicates.  This will highlight the weights around peaks
// since the diff between A/B near peaks is small compared to the diff in the 
// background.  avgYi is the average counts of pixel i between sides A/B.
//
// input:
//   iRow - DFMS side A/B
//
// modifies:
//   sets up new calculated values of weights
// returns;
//   None
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
void DFMS_PeakFit_Class::calcWeights(int iRow) {
    
    double *avg = new double[NUMPIX];
    for (int i=0; i<NUMPIX; ++i) {
        if (Yin[i] <= 0.0) {
            weight[i] = 0.0;
        } else {
             weight[i] = 1.0/sqrt(Yin[i]);
        }
    }
    
    delete[] avg;   avg=0;
    
}

//
// --------------------------- invertMatrix ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine inverts the provided N x N dimension matrix. A method
// called 'pivoting' reduces rounding error.
//
// inputs:
//   nDim - the value of N for the N x N dimension matrix
//
// modifies:
//   inArr - input array
//   the array of dimensions N x N which returns inverted
// returns:
//   None
// =============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
// =============================================================================
//
int DFMS_PeakFit_Class::invertMatrix(int nDim, double** inArr) {

	string sFunctionName = "DFMS_PeakFit_Class::invertMatrix";

	int ik[nDim];
	for (int i=0; i<nDim; i++) ik[i] = 0;

	int jk[nDim];
	for (int i=0; i<nDim; i++) jk[i] = 0;

	double nRsave = 0.0;;
	double nAmax = 0.0;

	for (int k=0; k<nDim; k++) {
		nAmax = 0.0;
		for (int i=k; i < nDim; i++) {
			for (int j=k; j < nDim; j++) {
				if (abs(nAmax) <= abs(inArr[i][j])) {
					nAmax = inArr[i][j];
					ik[k] = i;
					jk[k] = j;
				}
			}
		}

		int i = ik[k];
		if (i>k) {
			for (int j=0; j<nDim; j++) {
				nRsave = inArr[k][j];
				// Do a +/- Inf and NaN check. This is a C++11 std cmath function but
				// may not work on all compilers.
				//cerr << "inArr[" << i << "][" << j << "] = " << inArr[i][j] << endl;
				if (!isfinite(inArr[i][j])) {
					cout << "inArr[" << i << "][" << j << "] = " << inArr[i][j] << endl;
					sErrorMessage = "A C++ NaN was encountered. Possibly peak is too small for proper fit";
					writeToLog(sErrorMessage, sFunctionName, ERROR);
					infNanErr = true;
					return 0;
				}
				inArr[k][j] = inArr[i][j];
				inArr[i][j] = -1 * nRsave;
			}
		}

		int j = jk[k];
		if (j>k) {
			for (int i=0; i<nDim; i++) {
				nRsave = inArr[i][k];
				inArr[i][k] = inArr[i][j];
				inArr[i][j] = -1 * nRsave;
			}
		}

		for (int i=0; i<nDim; i++) {
			if (i != k)	inArr[i][k] = -1 * inArr[i][k] / nAmax;
		}

		for (int i=0; i<nDim; i++) {
			for (int j=0; j<nDim; j++) {
				if (j != k && i!= k) inArr[i][j] = inArr[i][j] + 
						   	   	   	 inArr[i][k] * inArr[k][j];
			}
		}

		for (int j=0; j<nDim; j++) {
			if (j != k)	inArr[k][j] = inArr[k][j]/nAmax;
		}
		inArr[k][k] = 1/nAmax;
	}
     
	for (int k=nDim-1; k>-1; k--) {
		int j = ik[k];
		if (j>k) {
			for (int i=0; i<nDim; i++){
				nRsave = inArr[i][k];
				inArr[i][k] = -1 * inArr[i][j];
				inArr[i][j] = nRsave;
			}
		}

		int i = jk[k];
		if (i>k) {
			for (j=0; j<nDim; j++) {
				nRsave = inArr[k][j];
				inArr[k][j]= -1 * inArr[i][j];
				inArr[i][j] = nRsave;
			}
		}
	}

	return 1;
}

//
// ------------------------------- curveFit ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine performs a fit of a 2D (X,Y) data set to the function
// set in "curveFit fitFunc" by adjusting an initial set of parameter values
// using the Marquardt algorithm.  The routine iterates until an improvement in 
// the chi-squared statistic is achieved from the initial parameters.
//
// inputs:
//   Yin - the array of Y-coordinate data values
//   weight - the array of weights of each data point to fit
//
// modifies:
//   lambda - returns updated depending on the success or failure of the
//              performed iteration in finding a better chi-squared statistic
//              by changing the values of the function parameters (returns
//              divided by 10 if better, unmodified if worse)
//   coeff - returns with updated values of the function parameters
//   inArr - returns the updated Array matrix from the curvefit procedure
//   derivs - returns the updated Derivative matrix from the curvefit procedure
//   alpha - returns the updated Alpha matrix from the curvefit procedure
//   beta - returns the updated Beta vector from the curvefit procedure
//  returns:
//   fitChiSquare - the calculated chi-squared statistic of the final iteration
// =============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
// =============================================================================
//
double DFMS_PeakFit_Class::curveFit(double* Yin, double* lambda, double** inArr,
                                  double** derivs, double** alpha, double* beta) 
{

	string sFunctionName = "DFMS_PeakFit_Class::curveFit";
	
	double chiSq;
	double testChiSquare;
	int nDegree;

	double *testCoeff = new double[nParams];

	// 2.4.2.1 - Define the degrees of freedom
	nDegree = NUMPIX - nParams;

	// 2.4.2.2 - Initialize the test parameters and Beta vector to zero
	//  and Alpha matrix values left of diagonal to zero
	for (int j=0; j<nParams; j++) {
		testCoeff[j] = 0;
		beta[j] = 0;
		for (int k=0; k<=j; k++) alpha[j][k] = 0;
	}

	// 2.4.2.3 - Calculate the initial fitted data array yFit
	for (int i=0; i<NUMPIX; i++) {
		yFit[i] = fitFunc(Xin[i], coeff);
		//cout << "yFit[" << i << "] = " << yFit[i] << endl;
	}

	// 2.4.2.4 - Calculate chisquare
	chiSq = chiSquare(NUMPIX, weight, Yin, yFit, nDegree);
	testChiSquare = chiSq;
	calcDeriv(derivs);

	// 2.4.2.5 - Update the Alpha matrix and Beta vector
	for (int i=0; i<NUMPIX; i++) {
		for (int j=0; j<nParams; j++) {
			beta[j] += weight[i] * (Yin[i] - yFit[i]) * derivs[i][j];
			for (int k=0; k<=j; k++) {
				alpha[j][k] += (weight[i] * derivs[i][j] * derivs[i][k]);
			}
		}
	}

	// 2.4.2.6 - Finish modifications of the Alpha matrix
	for (int j=0; j<nParams; j++) {
		for (int k=0; k<=j; k++) alpha[k][j] = alpha[j][k];
	}

	// 2.4.2.7 - Test fits using the test parameters (modifying them each iteration) until their
	//  fit has an improved chi-squared statistic
	do {
          for (int j=0; j<nParams; j++) {
        	  for (int k=0; k<nParams; k++) {
        		  inArr[j][k] = alpha[j][k] / sqrt(alpha[j][j] * alpha[k][k]);
  				  //cout << "alpha[" << j << "][" << k << "] = " << alpha[j][k] << endl;
              }
        	  inArr[j][j] = *lambda + 1;
          }

          int nSuccess = invertMatrix(nParams, inArr);
          if (!nSuccess) {
				skipReas["curvFitBad"]++;
        	  sErrorMessage = "InvertMatrix returned badly";
        	  writeToLog(sErrorMessage, sFunctionName, ERROR);
        	  return -1.0;
          }
          for (int j=0; j<nParams; j++) {
        	  testCoeff[j] = coeff[j];
        	  for (int k=0; k<nParams; k++) {
        		  testCoeff[j] += beta[k] * inArr[j][k] /
        				          sqrt(alpha[j][j] * alpha[k][k]);
        	  }
          }

          for (int i=0; i<NUMPIX; i++) {
        	  yFit[i] = fitFunc(Xin[i], testCoeff);
          }

          chiSq = chiSquare(NUMPIX, weight, Yin, yFit, nDegree);
          if ((testChiSquare - chiSq) < 0) *lambda *= 10;

	} while(chiSq > testChiSquare);


	// 2.4.2.8 - Set the parameter values to the improved test values
	for (int j=0; j<nParams; j++) coeff[j] = testCoeff[j];
    
    // define fwhm (initially set to a constant in constructor.
    fwhm = 2.354820045*coeff[2];

	*lambda /= 10;

	// 2.4.2.9 - Release memory
	delete[] testCoeff; testCoeff=0;

	return chiSq;
}

//
// --------------------------- processCurveFit ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine performs the non-linear curvefit on a two-dimensional data set
// to the function set in "BAM_curvefit_fit_func", using the Marquardt algorithm.
// The routine iterates until the reduced chi-squared statistic of consecutive
// iterations is less than the provided threshold ("fitChiSq_diff_percent_thresh")
// or the number of itera rises beyond the provided threshold ("maxIterations").
// The Marquardt alrogithm is implemented as described in P.R. Bevington, "Data
// Reduction and Error Analysis for the Physical Sciences," McGraw-Hill, 1969;
// Adapted by Wayne Weimer & David Harris.  Translated to ANSI C by Doug. Harris
// & Tim Seufert 7-94.  Modified by Brian Magee in March 2012 for user
// adaptability and portability
//
// inputs (some are not passed but are class variables):
//  iRow     - Side A or B of the input data
//   maxIters - maximum number of iterations to perfrom
//   weight   - Class-var, the weight of each data point to provide the 
//              curvefit procedure
//   coeff    - Class-var, the number of parameters in the fitting function
//   fitChiSq_diff_percent_thresh - the percentage difference between the chi-
//           squared statistic of consecutive fits to meet in order to determine  
//           a suitable stopping point in the curveFit procedure
//   maxIterations - the maximum number of iterations to perform before ejecting 
//           from the curvefit procedure
//   printIters - Flag to print iteration info
//
// modifies: 
//   coeff    - returns as the parameters of the final iteration of the  
//              curvefit procedure
//   nIters   - returns as the number of iterations performed before  
//              curvefit completed or maximum number of iterations achieved
//   lambda   - returns as the final Flambda of the last iteration of the  
//              curvefit procedure
//   fitChiSq - returns as the final achieved fit chi-squared statistic
//   fitChiSqHistory - returns filled with the history of chi-squared values 
//                     by iteration.
// returns:
//   1 for a successful run, 0 to indicate an error to be handled
// =============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
// =============================================================================
//
int DFMS_PeakFit_Class::processCurveFit(int iRow, int printIters)
{

	string sFunctionName = "DFMS_PeakFit_Class::processCurveFit";
	
	int nIters = 0;
	int maxIters = MAXITERS;
	double lambda = 0.0;
	double fitChiSq = 0.0;
	double *fitChiSqHistory = new double[maxIters];
	double fitChiSq_diff_percent_thres = pow(10.0,-8);


	// 2.4.1 - Allocate memory to needed variables and set variables
	double fitChiSq_former;
	double fitChiSq_diff_percent;
     
	double* anB = new double[nParams];
	double* anC = new double[nParams];
	double* beta = new double[nParams];

	// Create [nParams x nParams] array
	double **alpha = new double*[nParams];
	for(int i=0; i<nParams; i++){
		alpha[i] = new double[nParams];
	}
 
	// Create [nParams x nParams] array
	double **inArr = new double*[nParams];
	for(int i=0; i<nParams; i++){
		inArr[i] = new double[nParams];
	}
 
	// Create [NUMPIX x nParams] array
	double **derivs = new double*[NUMPIX];
	for(int i=0; i<NUMPIX; i++ ){
		derivs[i] = new double[nParams];
	}
     
	// Create input Y array from side info
	double *Yinp = new double[NUMPIX];
	for(int i=0; i<NUMPIX; ++i) {
		Yinp[i] = Yin[i];
		if (verbose == 100) cout << "processCurveFit - Yinp[" << i << "] = " << Yinp[i] << endl;
	}
     
	// Set the values for the needed variables
	for (int i=0; i<nParams; i++) {
		anB[i] = 0;
		anC[i] = 0;
		beta[i] = 0;

		for (int j=0; j<nParams; j++) {
			alpha[i][j] = 0;
			inArr[i][j] = 0;
		}
	}

	// 2.4.2 - Calculate the L2 Data weights used by the curvefit method
	// TODO - fix numbering
	calcWeights(iRow);

	for (int i=0; i<NUMPIX; i++) {
		for (int j=0; j<nParams; j++) derivs[i][j] = 0;
	}
	fitChiSq_former = pow(10.0,99);

	//cout << "Beginning of processCurvfit: coeff[1] = " << coeff[1] << endl;

	// 2.4.3 - Perform the curve fitting until the chi-square value for
	//         consecutive iterations changes by less than 10^-7 percent
	do {
		fitChiSq = curveFit(Yinp, &lambda, inArr, derivs, alpha, beta);
		if (fitChiSq == -1.0) {
	    	 sErrorMessage = "CurveFit returned badly";
	    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
	    	 return 0;
		}

		fitChiSqHistory[nIters] = fitChiSq;

		nIters = nIters + 1;
		fitChiSq_diff_percent = fabs(fitChiSq_former - fitChiSq)/fitChiSq_former;
		fitChiSq_former = fitChiSq;

		if (printIters == 1) {
			printIteration(nIters, fitChiSq, fitChiSq_diff_percent,
                              beta, alpha, inArr, derivs);
		}
	}
	while (fitChiSq_diff_percent > fitChiSq_diff_percent_thres && nIters < maxIters);

	//cout << "End of processCurvfit: coeff[1] = " << coeff[1] << endl;

	// 2.4.4 - Free the allocated memory no longer needed
	delete[] Yinp;    Yinp = 0;
	delete[] anB;    anB = 0;
	delete[] anC;    anC = 0;
	delete[] beta;   beta = 0;
	for (int i = 0; i < nParams; i++) delete[] alpha[i];
	delete[] alpha;  alpha =0;
	for (int i = 0; i < nParams; i++) delete[] inArr[i];
	delete[] inArr;  inArr = 0;
	for (int i = 0; i < NUMPIX; i++) delete[] derivs[i];
	delete[] derivs; derivs = 0;
	delete[] fitChiSqHistory; fitChiSqHistory=0;

	// 2.4.5 - Set error and warning flags
	if (fitChiSq_diff_percent > fitChiSq_diff_percent_thres || nIters >= maxIters) {
		    if (fitChiSq_diff_percent > fitChiSq_diff_percent_thres) {
				sErrorMessage = "CurveFit completed with unacceptable ChiSquare";
				writeToLog(sErrorMessage, sFunctionName, ERROR);
				if (verbose >= 3) cout << "processCurveFit - fitChiSq_diff_percent = " << fitChiSq_diff_percent << endl;
		    } else if (nIters >= maxIters) {
				sErrorMessage = "Iteration > Max Iterations";
				writeToLog(sErrorMessage, sFunctionName, ERROR);
				if (verbose >= 3) cout << "processCurveFit - *nIters = " << nIters << endl;
		    }

			return 0;
	}

	return 1;
}

//
// ------------------------- printIteration ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine prints to screen or to file the results of a single iteration in 
// the curvefit algorithm.
//
// inputs:
//   //   nIters - the number of this iteration
//   fitChiSquare - the value of the chi-squared statistic of this iteration
//   fitChiSq_diff_percent - the value of the difference between the chi-squared
//                                statistic of this iteration and the previous
//   nParams - the number of parameters in the fit function
//   NUMPIX - the number of records of the data arrays
//   coeff - the function parameters resulting from this iteration
//   beta - the Beta vector used in the curvefit algorithm at this iteration
//   alpha - the Alpha matrix used in the curvefit algorithm at this iteration
//   inArr - the Array matrix used in the curvefit algorithm at this iteration
//   derivs - the Derivative matrix used in the curvefit algorithm at this iteration
//
// returns:
//   None
// =============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
// =============================================================================
//
void DFMS_PeakFit_Class::printIteration (int nIter, double fitChiSq,
               double fitChiSq_diff_percent, double *beta, double **alpha, 
               double **inArr, double **derivs) {

	string sFunctionName = "DFMS_PeakFit_Class::printIteration";
     
	printf("\n===============");
	printf("\n iteration %d", nIter);
	printf("\n===============");
	printf("\n\n Fit Chi Square = %f", fitChiSq);
	printf("\n Fit Chi Square Diff Percent = %3.2E", fitChiSq_diff_percent);


	printf("\n coeff = [");
	for (int i=0; i<nParams; i++) {
		printf("%f", coeff[i]);
		if (i<nParams-1) printf(", ");
	}
	printf("]");

	if (0) {
		printf("\n\n beta = [");
		for (int i=0; i<nParams; i++) {
			printf("%f", beta[i]);
			if (i<nParams-1) printf(", ");
		}
		printf("]");

		printf("\n\n alpha = [");
		for (int i=0; i<nParams; i++) {
			for (int j=0; j<nParams; j++) {
				printf("%f", alpha[i][j]);
				if (j<nParams-1) printf(", ");
			}
			if (i<nParams-1) {
				printf("\n            ");
			} else {
				printf("]");
			}
		}


		printf("\n\n inArr = [");
		for (int i=0; i<nParams; i++) {
			for (int j=0; j<nParams; j++) {
				printf("%f", inArr[i][j]);
				if (j<nParams-1) printf(", ");
			}
			if (i<nParams-1) {
				printf("\n            ");
			} else {
				printf("]");
			}
		}

		printf("\n\n derivs = [");
		for (int i=0; i<NUMPIX; i++) {
			for (int j=0; j<nParams; j++) {
				printf("%f", derivs[i][j]);
				if (j<nParams-1) printf(", ");
			}
			if (i<NUMPIX-1) printf("\n                 ");
			else printf("]");
		}
	}
	printf("\n\n");

}

// ------------------------ errorOnFitParams -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Calculate the error on the Gaussian fit coefficients
//
// inputs:
//   x1 - the point where to calculate the error
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi, June 2013
// =============================================================================
//
void DFMS_PeakFit_Class::errorOnFitParams(double x1) {
    
    // Calculate error on fit coefficients
    double expPart = exp(-pow((coeff[1]-x1)/coeff[2],2.0));
    errCoeff[0] = expPart;
    errCoeff[1] = 2*coeff[0]*(x1-coeff[1])/pow(coeff[2],2.0) * expPart;
    errCoeff[2] = 2*coeff[0]*pow(coeff[1]-x1,2.0)/pow(coeff[2],3.0) * expPart;                
}

//
// ------------------------ calcParamErrors -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Calculate the errors on the fit parameters coeff using the covariance matrix
// method
//
// inputs:
//	iRow - DFMS side/A/B
//
// returns:
//   none
// =============================================================================
// History:  Written by Mike Rinaldi, June 2013
// =============================================================================
//
int DFMS_PeakFit_Class::calcParamErrors(int iRow) {

	string sFunctionName = "DFMS_PeakFit_Class::calcParamErrors";
    
     double sigma=0.0;
    
     // Allocate memory for [NUMPIX x nParams] array
     double **derivs = new double*[NUMPIX];
     for(int i=0; i<NUMPIX; i++ ){
         derivs[i] = new double[nParams];
     } 
     
          
     // Allocate and initialize the fisher matrix to hold partial derivs
     double **fisher = new double*[nParams];
     for(int i = 0; i < nParams; ++i) fisher[i] = new double[nParams];
     for(int i = 0; i < nParams; ++i) {
         for (int j = 0; j < nParams; ++j) {
             fisher[i][j] = 0.0;
         }
     }
     
     // Allocate and initialize the covariance matrix to hold inverse of fisher
     double **covarMatrix = new double*[nParams];
     for(int i = 0; i < nParams; ++i) covarMatrix[i] = new double[nParams];
     for(int i = 0; i < nParams; ++i) {
         for (int j = 0; j < nParams; ++j) {
             covarMatrix[i][j] = 0.0;
         }
     }
     
 	// The latest fit results needed by calcDerivs
     for (int i=0; i<NUMPIX; i++) {
        	 yFit[i] = fitFunc(Xin[i], coeff);
        	 if (iRow == 1) {
        		 yFitA[i] = yFit[i];
        		 //cout << "yFitA[" << i << "] = " << yFitA[i] << endl;
        	 } else {
        		 yFitB[i] = yFit[i];
        	 }
     }

     // Calculate the derivatives
     calcDeriv(derivs);

     // Build the Fisher and placeholder Covariance matrix
     for (int j=0; j<nParams; ++j) {
    	 for (int k=0; k<nParams; ++k) {
    		 for (int i=0; i<NUMPIX; ++i) {
    			 sigma = sqrt(Yin[i]);  // estimate error
    			 if (sigma > 0.0) {
    				 fisher[j][k] += derivs[i][j]*derivs[i][k]/(sigma*sigma);
    			 }
             }
    		 covarMatrix[j][k] = fisher[j][k];
         }
     }
     
     // Invert place holder Covariance matrix to get the actual one
     int nSuccess = invertMatrix(nParams, covarMatrix);
     if (!nSuccess) {
    	 sErrorMessage = "InvertMatrix returned badly";
    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
    	 return 0;
     }
     
     // Now the coefficient errors are the diagonal elements of the Covariance
     // Matrix
     for (int j=0; j<nParams; ++j) {
         errCoeff[j] = sqrt(covarMatrix[j][j]);
     }
     // Override current method to calculate error on peak Location.
     // Simply use the peak width since this is the standard deviation
     // in the Gaussian fit.
     errCoeff[1] = coeff[2]/2.0;
    // cout << "coeff[0] = " << coeff[0] << endl;
    // cout << "coeff[1] = " << coeff[1] << endl;
    // cout << "coeff[2] = " << coeff[2] << endl;
     
     // Check on errCoeff and see if reasonable
     // Recall, coeff[0] = pkHght, coeff[1] = peak Location, coeff[2] = pk width
     // Assume coeff[0] < 1.0e10  therefore error < 1.0e5
     // Assume coeff[1] > 20 and < 490  therefore  error < 20
     // Assume coeff[2] < 100 therefore error < 10
     bool reasonable = true;
     if (errCoeff[0] > 1.0e5) reasonable = false;
     if (errCoeff[1] > 20) reasonable = false;
     if (coeff[2] <= 0 || errCoeff[2] > 10) reasonable = false;

     //cout << "errCoeff[0] = " << errCoeff[0] << endl;
     //cout << "errCoeff[1] = " << errCoeff[1] << endl;
     //cout << "errCoeff[2] = " << errCoeff[2] << endl;

     if (!reasonable) {
    	 sErrorMessage =  "Error on final curvefit coefficients are not reasonable assume bad fit";
    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
         // Release memory
         for (int i = 0; i < NUMPIX; i++) delete[] derivs[i];
         for (int j=0; j<nParams; ++j) {
             delete[] fisher[j];
             delete[] covarMatrix[j];
         }
         delete[] derivs; derivs = 0;
         delete[] fisher; fisher = 0;
         delete[] covarMatrix; covarMatrix = 0;
         return 0;
     }
    
     // Release memory
     for (int i = 0; i < NUMPIX; i++) delete[] derivs[i];
     for (int j=0; j<nParams; ++j) {
         delete[] fisher[j];
         delete[] covarMatrix[j];
     }
     delete[] derivs; derivs = 0;
     delete[] fisher; fisher = 0;
     delete[] covarMatrix; covarMatrix = 0;

     return 1;
    
}

//
// ------------------------ bestPeak -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Create the best Fit array
//
// inputs:
//	iRow - DFMS row A/B
//
// returns:
//   none
// =============================================================================
// History:  Written by Mike Rinaldi, June 2013
// =============================================================================
//
int DFMS_PeakFit_Class::bestPeak(int iRow) {

	// The latest fit results needed by calcDerivs
    for (int i=0; i<NUMPIX; i++) {
       	 yFit[i] = fitFunc(Xin[i], coeff);
       	 if (iRow == 1) {
       		 yFitA[i] = yFit[i];
       		 //cout << "yFitA[" << i << "] = " << yFitA[i] << endl;
       	 } else {
       		 yFitB[i] = yFit[i];
       	 }
    }

    return 1;
}

//
// ------------------------- checkHistoSmoothness ------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Check the resultant data histogram for smooth peak or edge issues.
//
// inputs:
//	hPeak - The histogram object
//
// returns:
//   true if data is conducive to peak fitting
//   false if data is not conducive to peak fitting
// =============================================================================
// History:  Written by Mike Rinaldi, June 2013
// =============================================================================
//
bool DFMS_PeakFit_Class::checkHistoSmoothness(DFMS_Histogram_Class *hPeak, int iRow) {

	string sFunctionName = "DFMS_PeakFit_Class::checkHistoSmoothness";

	//double pkHeight = pow(10.0,hPeak->getHmax());  // Peak height
	double pkLoc = hPeak->getImax()+iXl;      	// Peak (+Xl because array is offset)
	double pkWidth = 10.0;                    	// FWHM (use 10.0 as a reasonable guess)

	bool proceed = true;
	// Check if peak is too close to the edges
	if (pkLoc < pkWidth || pkLoc > NUMPIX-pkWidth) {
		proceed = false;    // Peak Location is too close to the edge
		sErrorMessage = "Peak Location is to close to the edge ";
		sErrorMessage += "[pkLoc < pkWidth OR pkLoc > 512-pkWidth]";
		writeToLog(sErrorMessage, sFunctionName, WARN);
	}
	// Check if dynamic range between offset and peak is too large.  This is not
	// an exact check but assumes well behaved peaks are not too high above Offset.
	// This is a subjective assumption attempting to identify step function-like
	// corrupted data
	double dLow=0.0;
	if (threshold < 0) {
		dLow = -threshold;
	} else {
		dLow = threshold;
	}
	double dHigh = hPeak->getHmax()+0.5;

	// TODO - Make 3.5 a Configuration variable?
	if (dHigh-dLow > 3.5) {
		proceed = false;    // possible step function
		if (iRow == 1) {
			stepFuncA = true;
		} else {
			stepFuncB = true;
		}
		sErrorMessage = "Data is likely corrupt [Peak Height/Offset > 10^3.5]";
		writeToLog(sErrorMessage, sFunctionName, WARN);
	}

	return proceed;

}
