/*
 * DFMS_PixelGain_Interp_Class.cc
 *
 *  Created on: Jul 30, 2015
 *      Author: marinaldi
 */

#include "DFMS_PixelGain_Interp_Class.hh"

//
// ---------------------------- Constructor -------------------------------------
//
DFMS_PixelGain_Interp_Class::DFMS_PixelGain_Interp_Class(string pgpath) {

	path = pgpath;

	for (int i=0; i< NUMPIX; i++) {
	    for (int l=0; l<5; l++) {
	    	pgFit[i][l] = 0.0;
	    }
	}

}

//
// ---------------------------- Destructor -------------------------------------
//
DFMS_PixelGain_Interp_Class::~DFMS_PixelGain_Interp_Class() {
	// TODO Auto-generated destructor stub
}

//
// ----------------------- findMulti_GS_PGT -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method finds all Pixel Gain Tables of a given Gain Step.  It sets the choose
// variable to true for all matching Gain Steps.
//
// inputs:
//   PGTtimeInfo   -  Vector with PGT file info
//   GS  -  Gain Step to search for.
//
// returns:
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, July 2015
// =============================================================================
//
int DFMS_PixelGain_Interp_Class::findMulti_GS_PGT(int GS, vector<pgFileInfo> &PGTinfo,
												vector<pgFileInfo> &P) {

	string sFunctionName="DFMS_PixelGain_Interp_Class::findMulti_GS_PGT";

	int numTrue=0;
	P.clear();
	for (int i=0; i<PGTinfo.size(); i++) {
		if (PGTinfo[i].GainStep == GS) {
			P.push_back(PGTinfo[i]);
			//P[numTrue].printHeader();
			//P[numTrue].printPG();
			numTrue++;
		}
	}

	if (numTrue <= 1) {
		sInfoMessage="Number of available Gain Steps for GS = "+util_intToString(GS);
		sInfoMessage+=" is <= to 1.  Cannot interpolate. Use closest/GS/Time method ";
		writeToLog(sInfoMessage, sFunctionName, INFO);
	}

	return numTrue;
}

//
// ---------------------------- readAllPGT -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method reads all the Pixel Gain file and stores in the data in the
// vector structure PGTinfo.
//
// inputs:
//   PGTinfo   -  Vector with PGT info
//
// returns:
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, July 2015
// =============================================================================
//
int DFMS_PixelGain_Interp_Class::readAllPGT(vector<pgFileInfo> &PGTinfo) {

	string sFunctionName="DFMS_PixelGain_Interp_Class::readAllPGT";
    
	// Now read in all Pixel Gain Tables
	for (int i=0; i<PGTinfo.size(); i++) {
		string pgpath = path+PGTinfo[i].dirName+"/";
		string pgfile = PGTinfo[i].filename;
		DFMS_PixelGain_Table_Class *pixGTObject = new DFMS_PixelGain_Table_Class(pgpath, pgfile);
		int nSuccess = pixGTObject->getPGTable(pgpath, pgfile);
		if (!nSuccess) {
			sErrorMessage = "Reading Pixel Gain Table: " + pgfile+" was unsuccessful ";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			//cout << "Reading Pixel Gain Table: " << pgfile << " was unsuccessful" << endl;
			exit(EXIT_FAILURE);
		}
		PGTinfo[i].readPG(pixGTObject->pixGTdata);
		if (verbose >=3) PGTinfo[i].printHeader();
		if (verbose >=3) PGTinfo[i].printPG();
        
		delete pixGTObject;
	}

	return 1;
}

//
// -------------------------- dumpPGIntpData -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method will dump the interpolation results versus input data to check
// for accuracy
//
// inputs:
//  t   -  time when PG Table is required (in seconds since 19700101T00:00:00.00)
//  GS  -  the current GS value
//  P   -  The Pixel Gain info structure
//  pgO -  The pixel Gain Object
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, October 2015
// =============================================================================
//
void DFMS_PixelGain_Interp_Class::dumpPGIntpData(double t, int GS, vector<pgFileInfo> &P,
													DFMS_PixelGain_Table_Class *pgO) {

	double pgout[512][5][5] = {0};
	double pgTime[5] = {0};
	string pgFname[5];

	fstream out;
	string line;
	string ofile = "pixTest_GS"+util_intToString(GS)+".out";

	for (int k=0; k<P.size(); k++) {
		string sTime = P[k].dateTime+"_000000000";
		pgTime[k] = rosToSec(sTime, 1)/JULIANYEAR;
		pgFname[k] = P[k].filename;
		for (int i=0; i<NUMPIX; ++i) {
			pgout[i][0][k] = i+1;
			for (int j=1; j<5; j++) {
				pgout[i][j][k] = P[k].pixG[i][j];
			}
		}
	}

	// Set up I/O to create dump file
	out.open(ofile.c_str(),ios::out);
	out << "GS = " << GS << dfmsEOL;
	out << "t = " << t/JULIANYEAR << dfmsEOL;
	for (int i=0; i<5; i++) {
		out << "PG file: " << pgFname[i] << "  -  t[" << i << "]= " << pgTime[i] << dfmsEOL;
		cout << "PG file: " << pgFname[i] << "  -  t[" << i << "]= " << pgTime[i] << endl;
	}

	// Dump interpolated data at at time t and other PG data used for interpolation
	for (int i=0; i<NUMPIX; ++i) {
		pgFit[i][0] = i+1;
		for (int j=1; j<5; j++) {
			pgFit[i][j] = pgO->pixGTdata[i][j];
		}
		char tmp[200];
		int pix = (int)pgout[i][0][0];
		sprintf(tmp,"%3.3d     %f  %f  %f  %f  %f  %f\n",
				pix,pgFit[i][1],pgout[i][1][0],pgout[i][1][1],pgout[i][1][2],pgout[i][1][3],pgout[i][1][4]);
		line.assign(tmp);
		//cout << line << endl;
		out << line;
	}
	out.close();
	cout << "Complete.... Check for pixTest.out file" << endl;
	exit(EXIT_SUCCESS);

}

//
// ---------------------------- buildPixGTdata -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method creates the required pixGTdata array using interpolation or
// extrapolation of relevant Pixel Gain tables
//
// inputs:
//   pgO -  The pixel Gain Table class object
//   GS	 -  The gain step
//   t   -  time when PG Table is required (in seconds since 19700101T00:00:00.00)
//
// returns:
//   returns 1 if interpolation was successful, 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, October 2015
// =============================================================================
//
int DFMS_PixelGain_Interp_Class::buildPixGTdata(DFMS_PixelGain_Table_Class *pgO,
		                           int GS, double t, vector<pgFileInfo> &P) {

	// Number of PG tables with this GS
	int num = P.size();

	// Calculate the scaled time of the L2 file
	double tL2 = t/JULIANYEAR;

	// Allocate memory to pixGTdata 2D vector array field variable and initialize to 0.0
 	for (int j=0; j<NUMPIX; j++) {
 		pgO->pixGTdata.push_back(dslice());
 		for (int i=0; i<5; i++) {
 			pgO->pixGTdata[j].push_back(0.0);
		}
	}

 	// Find time boundaries for these PG Tables
 	string bTime = P[0].dateTime+"_000000000";
 	double tb = rosToSec(bTime, 1)/JULIANYEAR;
 	string fTime = P[num-1].dateTime+"_000000000";
 	double tf = rosToSec(fTime, 1)/JULIANYEAR;

 	// Find how to interpolate
 	// if tL2 > tf:  Extrapolate Forwards using the last two PG tables for this GS
 	// if tL2 < tb:  Extrapolate backwards using the first two PG tables for this GS
 	// if tb <= tL2 <= tf:  Interpolate using the two time bracketing PG tables for this GS

// 	for (int i=0; i<num; ++i) {
// 		cout << i << " - P date: " << P[i].dateTime << endl;
// 	}

 	if (tL2 > tf) {   // ----------------------------  Extrapolate Forwards using the last two PG tables for this GS
        //cout << "Gain Step = " << P[num-2].GainStep << endl;
 		string sTime = P[num-2].dateTime+"_000000000";
 	 	//cout << "sTime = " << sTime << endl;
 		double ts = rosToSec(sTime, 1)/JULIANYEAR;
 		string eTime = P[num-1].dateTime+"_000000000";
 	 	//cout << "eTime = " << eTime << endl;
 		double te = rosToSec(eTime, 1)/JULIANYEAR;
 		for (int i=0; i<NUMPIX; ++i) {
 			pgO->pixGTdata[i][0] = i+1;   // Just copy the pixel number
 			for (int j=1; j<5; j++) {
 				//P[num-2].printPG();
 				double Ps = P[num-2].pixG[i][j];
 				double Pe = P[num-1].pixG[i][j];
 				double Pt = Ps + (Pe-Ps)/(te-ts)*(tL2-ts);
// 				if (j == 3) {
// 					//cout << "[" << i << "][" << j << "]: Pt = " << Pt << endl;
// 					cout << i+1 << ", " << Ps << ", " << Pe << ", " << Pt << endl;
// 				}
 				pgO->pixGTdata[i][j] = Pt;
		 	}
		 }
 		//exit(EXIT_SUCCESS);

 	} else if (tL2 < tb) {   // ----------------------  Extrapolate Backwards using the first two PG tables for this GS

 		string sTime = P[0].dateTime+"_000000000";
 		double ts = rosToSec(sTime, 1)/JULIANYEAR;
 		string eTime = P[1].dateTime+"_000000000";
 		double te = rosToSec(eTime, 1)/JULIANYEAR;
 		for (int i=0; i<NUMPIX; ++i) {
 			pgO->pixGTdata[i][0] = i+1;   // Just copy the pixel number
 			for (int j=1; j<5; j++) {
 				double Ps = P[num-2].pixG[i][j];
 				double Pe = P[num-1].pixG[i][j];
 				double Pt = Ps + (Pe-Ps)/(te-ts)*(tL2-ts);
 				pgO->pixGTdata[i][j] = Pt;
		 	}
		}


 	} else if (tL2 >= tb && tL2 <= tf) {   // ----------  Interpolate using the time bracketing PG tables for this GS

 		for (int k=0; k<P.size()-1; k++) {
 			string sTime = P[k].dateTime+"_000000000";
 		 	double ts = rosToSec(sTime, 1)/JULIANYEAR;
 		 	string eTime = P[k+1].dateTime+"_000000000";
 		 	double te = rosToSec(eTime, 1)/JULIANYEAR;
 		 	if (tL2 >= ts && tL2 <= te) {
 		 		for (int i=0; i<NUMPIX; ++i) {
 		 			pgO->pixGTdata[i][0] = i+1;   // Just copy the pixel number
 		 			for (int j=1; j<5; j++) {
 		 				double Ps = P[k].pixG[i][j];
 		 				double Pe = P[k+1].pixG[i][j];
 		 				double Pt = Ps + (Pe-Ps)/(te-ts)*(tL2-ts);
 		 				pgO->pixGTdata[i][j] = Pt;
 		 			}
 		 		}
 		 	}
 		}

 	} else {
 		if (L3INFOTOFILE) l3Log << "Unknown error with L2 time during PG interpolation" << dfmsEOL;
 		cout << "Unknown error with L2 time during PG interpolation" << endl;
 		return 0;
 	}

 	// If necessary dump the Pixel Gain data and resulting interpolation to
 	// check for accuracy.  Must set DUMPPGINTP == 1 in file DFMS_definedConstants.hh
 	// This is used mainly for interpolation diagnostics and thus will dump
 	// One GS value (set in if block below) then exit.
 	if (DUMPPGINTP && GS == 12) {
 		dumpPGIntpData(t, GS, P, pgO);
 	 	exit(EXIT_SUCCESS);
 	}

	return 1;
}
