
#include "DFMS_PixelGain_Table_Class.hh"

//
// ---------------------------- Constructor 1 ------------------------------------
//
DFMS_PixelGain_Table_Class::DFMS_PixelGain_Table_Class(string p, string f) {
   
	path = p;
	fileName = f;
	TableRows = 0;
	cols = 0;

}

//
// ---------------------------- Constructor 2 ------------------------------------
//
DFMS_PixelGain_Table_Class::DFMS_PixelGain_Table_Class() {

	TableRows = 0;
	cols = 0;

}
//
// ---------------------------- Destructor ------------------------------------
//
DFMS_PixelGain_Table_Class::~DFMS_PixelGain_Table_Class() {
    
	// Release dynamically allocated memory
	pixGTobj.clear();
	pixGTdata.clear();
}

//
// ------------------------------- getPGTable ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method reads a specific pixel gain table and stores the data in the
// members of the pixGTObject
//
// inputs:
//   path  	-  path to pixel gain table
//   file   -  file name
//
// outputs:
//   pixGTObject   -  pixel gain object
//
// returns:
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, Feb 2014
//          August 2015 - Moved the function DFMS_get_PixelGain_Table into
//                        this method
// =============================================================================
//
int DFMS_PixelGain_Table_Class::getPGTable(string path, string file)
{

	string sFunctionName = "DFMS_PixelGain_Table_Class::getPGTable";

	int nSuccess = 0;
	string a;
	fstream is;
	int rows;

	// Open GT file
	openFile(path+file,is,1);

	// Read the Pixel Gain Table file PDS header
	nSuccess = readPDSheader(is);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the PDS header for file: ";
		sErrorMessage += file+"\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Read the Pixel Gain Table file GT object description
	nSuccess = readObj(is, pixGTobj, rows, cols);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the pixGT Object Descp. for file: ";
		sErrorMessage += file+"\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Read the Gain Table file Data
	nSuccess = readpixGTdata(is, rows);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the pixGT Table file: ";
		sErrorMessage += file+"\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	// Close the GT File stream
	closeFile(is);

	sInfoMessage = "Completed Reading Pixel Gain Table File: ";
	sInfoMessage += file;
	writeToLog(sInfoMessage, sFunctionName, INFO);

	if (verbose >= 3) {
		cout << "completed Reading Pixel Gain Table " << file << endl;
    }


	return 1;
}


//
// ---------------------- readpixGTdata -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method retrieves all the Pixel Gain time sub-directories in a directory
//
// inputs:
//   is  	-  input stream
//
// outputs:
//   Rows   -  Number of rows read
//
// returns:
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, Feb 2014
// =============================================================================
//
int DFMS_PixelGain_Table_Class::readpixGTdata(fstream &is, int &Rows) {

	string sFunctionName = "DFMS_PixelGain_Table_Class::readpixGTdata";

	char input[100];

	TableRows = Rows;

	//cout << "Reading Pixel Gain Table File: " << fileName << endl;
	if (L3INFOTOFILE && verbose >= 3) l3Log << "Using Pixel Gain Table File: " << fileName << dfmsEOL << dfmsEOL;

	// Now read the pixel Gain table data into the data array
 	for (int i=0; i<TableRows; i++) {
         
 		pixGTdata.push_back(dslice());
          
 		for (int j=0; j<5; j++) {
 			is.getline(input,100,',');
			pixGTdata[i].push_back(atof(input));
			//cout << "pg[" << i << "][" << j << "] = " << pixGTdata[i][j] << endl;
		}
	}

	return 1;	
}

//
// ---------------------- printpixGTtable -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Print the interesting members of the pixelGainTable
//
// inputs:
//   type  -  Which part to print
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, Feb 2014
// =============================================================================
//
void DFMS_PixelGain_Table_Class::printpixGTtable(string type) {

	// First print the PDS Header
	
	mapstr::const_iterator it;

	if (type == "PDSHeader" || type == "All") {
		for (it = PDSheader.begin(); it != PDSheader.end(); ++it) {
			printf("%-32s =     %-s%s",(it->first).c_str(),(it->second).c_str(),EOL);
		}
	}

	// Next print out Object description
	if (type == "pixGTobj" || type == "All") {
		for (it = pixGTobj.begin(); it != pixGTobj.end(); ++it) {
			printf("%32s =     %-s%s",(it->first).c_str(),(it->second).c_str(),EOL);
		}
	}

	// Finally print out the Pixel Gain Table data
	if (type == "pixGTdata" || type == "All") {
	 	for (int i=0; i<TableRows; i++) {
	 		cout << pixGTdata[i][0] << " , ";
	 		cout << pixGTdata[i][1] << " , ";
	 		cout << pixGTdata[i][2] << " , ";
	 		cout << pixGTdata[i][3] << " , ";
	 		cout << pixGTdata[i][4] << " , " << endl;
		}	
	}
}

//
// ---------------------- getPGTtimeInDir -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method retrieves all the Pixel Gain time sub-directories in a directory
//
// inputs:
//   sPGTpath  -  Path to Pixel Gain tables
//   PGTtimeInfo   -  Vector with PGT file info
//
// returns:
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, Feb 2014
// =============================================================================
//
int DFMS_PixelGain_Table_Class::getPGTtimeInDir(string sPGTpath,
		                                       vector<DirInfo> &PGTtimeInfo) {

	string sFunctionName="DFMS_PixelGain_Table_Class::getPGTtimeInDir";

  	DIR *dir;
  	struct dirent *entry;
	string dirName;
	DirInfo D;
	PGTtimeInfo.clear();

	// Look into specified Instrument Model Directory
	//cout << "sPGTpath = " << sPGTpath << endl;

	if ((dir = opendir(sPGTpath.c_str())) == NULL) {
		sErrorMessage="Default Source directory - "+sPGTpath+" does not exist";
		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
  	} else {
  		while ((entry = readdir(dir)) != NULL) {
  			dirName = string(entry->d_name);
			// Skip all files/directories shorter than 3 characters.. (like ./, ../)
			if (dirName.size() > 3) {
				int dirNameSize = dirName.length();
				// Only use directory names of size 8 characters
				if (dirNameSize == 8) {
					D.dirName = util_trim(dirName);
					D.dateTime = util_trim(dirName);
					//D.printData();
					PGTtimeInfo.push_back(D);
				} else {
					sErrorMessage="Directory: "+dirName+" is not a candidate directory";
					writeToLog(sErrorMessage, sFunctionName, WARN);
					cout << sErrorMessage << endl;
				}
			}
  		}
  		closedir(dir);
  	}

	if (PGTtimeInfo.size() == 0) {
		sErrorMessage="Default source directory - " + sPGTpath+" does not contain expected files";
		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
	}

     return 1;
}

// Comparison function used by sort
bool myComp(string s1, string s2) { return (s1<s2); }

//
// ---------------------- getPGTFileInfo -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method retrieves all the Pixel Gain Table File information.
//
// inputs:
//   sPGTpath  -  Path to Pixel Gain tables
//   PGTFileInfo   -  Vector with PGT file info
//
// returns:
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, July 2015
// =============================================================================
//
int DFMS_PixelGain_Table_Class::getPGTFileInfo(string sPGTpath,
		                                       vector<pgFileInfo> &PGTFileInfo) {

	string sFunctionName="DFMS_PixelGain_Table_Class::getPGTFileInfo";

  	DIR *topDir, *pgDir;
  	string file;
  	struct dirent *entry;
	string dirName;
	pgFileInfo P;
	PGTFileInfo.clear();

	vector<string> dVec, fVec;  // Vector t ohold sorted dirs and files

	int numFile=0;
	if ((topDir = opendir(sPGTpath.c_str())) == NULL) {
		sErrorMessage="Default Source directory - "+sPGTpath+" does not exist";
		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
  	} else {
  		while ((entry = readdir(topDir)) != NULL) {
  		  		dirName = string(entry->d_name);
  				// Skip all files/directories shorter than 3 characters.. (like ./, ../)
  				if (dirName.size() == 8) {
  					dVec.push_back(dirName);
  				} else {
  					sErrorMessage="Directory: "+dirName+" is not a candidate directory";
  					writeToLog(sErrorMessage, sFunctionName, WARN);
  					cout << sErrorMessage << endl;
  				}
  		}
  		closedir(topDir);

  		if (dVec.size() == 0) {
  			sErrorMessage="No pixel Gain Directories were found";
  			writeToLog(sErrorMessage, sFunctionName, FATALERROR);
  			cout << sErrorMessage << endl;
  			return 0;
  		}

  		// Sort the directory names
  		sort(dVec.begin(), dVec.end());

  		for (int i=0; i<dVec.size(); i++) {
			string thisDir = sPGTpath+dVec[i]+"/";
			if ((pgDir = opendir(thisDir.c_str())) == NULL) {
				sErrorMessage="Directory - "+thisDir+" does not exist";
				writeToLog(sErrorMessage, sFunctionName, FATALERROR);
			} else {
				cout << "Accessing Pixel Gain file information from " << thisDir << endl;
				// Read in all PGT file names and get info
				while ((entry = readdir(pgDir)) != NULL) {
					file = string(entry->d_name);
					if (file.size() > 3 && file.substr(0,1).compare(".") != 0 && file.substr(22,2).compare("GS") == 0) {
						fVec.push_back(file);
					}
				}
				closedir(pgDir);

				if (fVec.size() == 0) {
				  	sErrorMessage="No pixel Gain files were found in directory "+ dVec[i];
				  	writeToLog(sErrorMessage, sFunctionName, WARN);
				  	cout << sErrorMessage << endl;
				} else {

					// Sort the file names in each dVec directory
					sort(fVec.begin(), fVec.end(), myComp);

					for( int j=0; j<fVec.size(); j++) {
						P.index = j+1;
						P.filename = fVec[j];
						P.dirName = dVec[i];
						P.dateTime = fVec[j].substr(8,8);
						P.dType = fVec[j].substr(17,1);
						P.ModelType = fVec[j].substr(19,2);
						P.GainStep = util_stringToInt(fVec[j].substr(24,2));
						P.printHeader();
						PGTFileInfo.push_back(P);
					}
					fVec.clear();
				}
			}
		}
	}

	if (PGTFileInfo.size() == 0) {
		sErrorMessage="Default source directory - " + sPGTpath+" does not contain expected files";
		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
	} else {
		cout << "Pixel Gain File Info was stored for ";
		cout << PGTFileInfo.size() << " files in " << dVec.size() << " directories" << endl;
	}

     return 1;
}

//
// ----------------------- findBestPGTtime -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method finds the closest pixel Gain Table file to the current
// L2 file date and time
//
// inputs:
//   sPGTpath  -  Path to Pixel Gain tables
//   sRefDateAndTime - Input date and time
//   PGTtimeInfo   -  Vector with PGT file info
//
// returns:
//   sBestMPST - best file name found
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, Feb 2014
// =============================================================================
//
int DFMS_PixelGain_Table_Class::findBestPGTtime(string path, string sRefDateAndTime,
		       int GSin, int *GSuse, string &dType, string &subDirName,
               vector<pgFileInfo> &PGTFileInfo)
{

	string sFunctionName="DFMS_PixelGain_Table_Class::findBestPGTtime";

	DIR *dir;
  	struct dirent *entry;
	string fileName;
	string sBestPGtime;

	vector<int> F;

	map<double,string> diffPGTime;
    vector<pgFileInfo> pSub;

	string nBestDateAndTime = "0";
	string nDateAndTime;
	string nThisDateAndTime;

	string type;   // Pixel Gain type.
	               // One of: "M" measured, "I" interpolated, "E" extrapolated

	// Initially set to PGT
	nDateAndTime = sRefDateAndTime.substr(0,8);
	sBestPGTtime = "0";

	//cout << "Using Instrument Model = " << sInstrumentModel << endl;
	if (L3INFOTOFILE && verbose >= 3) l3Log << "Using Instrument Model = " << sInstrumentModel << dfmsEOL;

	//
	// -------------- Look for Pixel Gain file closest in time
	//
	if (PGTFileInfo.size() > 0) {
		//cout << "PGTFileInfo size = " << PGTFileInfo.size() << endl;
		// Filter directory contents for those that match NG COPS Data
		// file info
		for (int i=0; i<PGTFileInfo.size(); i=i+1) {
			nThisDateAndTime = PGTFileInfo[i].dateTime+"_000000000";
			//cout << "sRefDateAndTime = " << sRefDateAndTime << endl;
			//cout << "nThisDateAndTime = " << nThisDateAndTime << endl;
			double diff = timeDiffInSec(nThisDateAndTime,sRefDateAndTime);
			//cout << "diff = " << diff << endl;
			sBestPGTtime = PGTFileInfo[i].dateTime;
			diffPGTime[diff] = sBestPGTtime;
		}
	}

	// No matching files found
	if (sBestPGTtime == "N/A") {
		sErrorMessage = "No suitable PG Data file ";
		sErrorMessage += "was found in supplied path: " + path;
		writeToLog(sErrorMessage, sFunctionName, WARN);
	} else {
		// Use C++ map automatic key sort property.
		// First element is the shortest time diff
		sBestPGTtime = diffPGTime.begin()->second;
        
        //Now find the directory name associated with this PG file date
        for (int i=0; i<PGTFileInfo.size(); i=i+1) {
            if (sBestPGTtime.compare(PGTFileInfo[i].dateTime) == 0) {
                subDirName = PGTFileInfo[i].dirName;
                break;
            }
        }
        
        // Check for some weird unforseen error
        if (subDirName.size() <= 1) {
            sErrorMessage = "Unknown Error:  This should never happen but .... ";
            writeToLog(sErrorMessage, sFunctionName, ERROR);
            return 0;
        }

		cout << "Found Best PG time: " << sBestPGTtime << endl;
	}

	//cout << "Nearest Pixel Gain Table date = " << sBestPGTtime << endl;
	if (L3INFOTOFILE && verbose >= 3) l3Log << "Nearest Pixel Gain Table date = " << sBestPGTtime << dfmsEOL;

	// We found a suitable time, now look for how many GS files are available
	// Then find the best Gain Step file
	if (sBestPGTtime != "0") {
		string sPath = path+subDirName+"/";
		if ((dir = opendir(sPath.c_str())) == NULL) {
			sErrorMessage="Default Source directory - "+sPath+" does not exist";
			writeToLog(sErrorMessage, sFunctionName, FATALERROR);
	  	} else {
	  		while ((entry = readdir(dir)) != NULL) {
	  			fileName = string(entry->d_name);
				// Skip all files shorter than 3 characters.. (like ./, ../)
	  			// eg., PIXGAIN_20100426_M_FS_GS13.TAB
	  			//cout  << "fileName = " << fileName << endl;
				if (fileName.size() > 3 && fileName.substr(0,1).compare(".") != 0 && fileName.substr(22,2).compare("GS") == 0) {
					string sGS = fileName.substr(22,4);
					//cout  << "GS = " << sGS << endl;
					F.push_back(util_stringToInt(sGS.substr(2,2)));
				}
	  		}
	  		closedir(dir);
	  	}
        
        // Sort the F Vector
        sort(F.begin(), F.end());

		// Find best GS
		map<int,int> FfromDiff;
		int thisOne = 0;
		if (F.size() >= 1) {
			if (GSin <= F[0]) {
				thisOne = F[0];
			} else if (GSin >= F[F.size()-1]) {
				thisOne = F[F.size()-1];
			} else if (GSin > F[0] && GSin < F[F.size()-1]) {
				for (int i=0; i<F.size()-1; i++) {
					if (GSin >= F[i] && GSin <= F[i+1]) {
						int diff1 = abs(F[i]-GSin);
						int diff2 = abs(F[i+1]-GSin);
						if (diff1 <= diff2) {
							FfromDiff.insert(pair<int,int>(diff1,F[i]));
						} else {
							FfromDiff.insert(pair<int,int>(diff2,F[i+1]));
						}
					}
				}
				// Since C++ maps order keys smallest to large, choose first element
				// which will necessarily be the smallest difference thus the nearest
				// gain step
				thisOne = FfromDiff.begin()->second;
			}
			cout << "Requested Gain Step: " << GSin << " - Available Gain Step: " << thisOne << endl;
			if (L3INFOTOFILE && verbose >= 3) l3Log << "Requested Gain Step: " << GSin << dfmsEOL;
			if (L3INFOTOFILE && verbose >= 3) l3Log << "Available Gain Step: " << thisOne << dfmsEOL;
			*GSuse = thisOne;
		} else {
			// Suitable GS not found
			sErrorMessage = "No suitable Gain Step file ";
			sErrorMessage += "was found in supplied path: " + path;
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			return 0;
		}

		FfromDiff.clear();

	} else {
	// Suitable time not found
		sErrorMessage = "No suitable Pixel Gain Table time ";
		sErrorMessage += "was found in supplied path: " + path;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
    
    // Now find dType
    for (int i=0; i<PGTFileInfo.size(); i=i+1) {
        if ( subDirName.compare(PGTFileInfo[i].dirName) == 0 )  {
            pSub.push_back(PGTFileInfo[i]);
        }
    }
    
    // Now check subdir for correct GS value
    for (int i=0; i<pSub.size(); i=i+1) {
        if (pSub[i].GainStep == 8) {
            dType = pSub[i].dType;
            sBestPGTtime = pSub[i].dateTime;
            break;
        }
    }
    
    // Check if dType is not defined
    if (dType == "") {
        sErrorMessage = "dType = " + dType + " is undefined" ;
        writeToLog(sErrorMessage, sFunctionName, ERROR);
        return 0;
    }

	return 1;

}
