/*
 * DFMS_X0FitFile_Class.cc
 *
 *  Created on: Jul 1, 2014
 *      Author: marinaldi
 */

#include "DFMS_X0FitFile_Class.hh"

//
// ------------------------ Reading Constructor -----------------------------
//
DFMS_X0FitFile_Class::DFMS_X0FitFile_Class(string p) {

	path = p;
	fitRowsA = 0;
	fitRowsB = 0;
	fitCols = 0;
	pixFitRows = 0;

	pix0FitFile = "None";
	fileName = "None";
	fileDate = "None";

	qualID = -1;

	aA=0.0, aB=0.0;
	bA=0.0, bB=0.0;
	sErrorA=0.0, sErrorB=0.0;
	RcoeffA=0.0, RcoeffB=0.0;
	R2coeffA=0.0, R2coeffB=0.0;
	errOnaA=0.0, errOnbA=0.0;
	errOnaB=0.0, errOnbB=0.0;

}

//
// ------------------------ Writing Constructor -----------------------------
//
DFMS_X0FitFile_Class::DFMS_X0FitFile_Class(string p, string f, int cRA, int cRB) {

	path = p;
	fileName = f;
	pix0FitFile = path+fileName;
	fitRowsA = cRA;
	fitRowsB = cRB;
	pixFitRows = 2;
	qualID = -1;

	fitCols = 6;

	// Allocate memory and initialize data arrays
	for(int i=0; i<fitRowsA; ++i) {
		fitDataA.push_back(sslice());
		for (int j=0; j<fitCols; ++j) {
			fitDataA[i].push_back("");
		}
	}

	// Allocate memory and initialize data arrays
	for(int i=0; i<fitRowsB; ++i) {
		fitDataB.push_back(sslice());
		for (int j=0; j<fitCols; ++j) {
			fitDataB[i].push_back("");
		}
	}

	aA=0.0, aB=0.0;
	bA=0.0, bB=0.0;
	sErrorA=0.0, sErrorB=0.0;
	RcoeffA=0.0, RcoeffB=0.0;
	R2coeffA=0.0, R2coeffB=0.0;
	errOnaA=0.0, errOnbA=0.0;
	errOnaB=0.0, errOnbB=0.0;

}

//
// ---------------------------- Destructor ---------------------------------
//
DFMS_X0FitFile_Class::~DFMS_X0FitFile_Class() {

	fitObjA.clear();
	fitObjB.clear();
	fitResDataObj.clear();
	fitDataA.clear();
	fitDataB.clear();

}

//
// ---------------------------- getFileName ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Return the unqualified pathname
//
// inputs:
//
// returns:
//   string - filename
// =============================================================================
// History:  Written by Mike Rinaldi, August 2014
// =============================================================================
string DFMS_X0FitFile_Class::getFileName() {

	//cout << "fileName = " << fileName << endl;
	unsigned found = fileName.find_last_of("/\\");
	string path = fileName.substr(0,found);
	string file = fileName.substr(found+1);

	return file;

}

//
// ------------------------------ readFitData ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Reads the fit data from the pix0 fit file
//
// inputs:
//  is  -  Reference to the input stream
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, July 2014
// =============================================================================
int DFMS_X0FitFile_Class::readFitData(fstream &is) {

	string sFunctionName = "DFMS_X0FitFile_Class::readFitData";

	char input[200];
	vector<string> fields;

	// Now read the fit data row A into the string array
 	for (int i=0; i<fitRowsA; i++) {

		// Use vector objects dynamic fill to create one row
 		fitDataA.push_back(sslice());

		is.getline(input,200);
		string line = string(input);
		//cout << "line = " << line << endl;
		util_splitString(line, ',', fields);
		for (int j=0; j<6; ++j) {
			//cout << "fields[" << j << "] = " <<  fields[j] << endl;
			fitDataA[i].push_back(fields[j]);
			//cout << "fitDataA[" << i << "][" << j << "] = "<<  fitDataA[i][j] << endl;
		}
        fields.clear();
	}

	// Now read the fit data row B into the string array
 	for (int i=0; i<fitRowsB; i++) {

		// Use vector objects dynamic fill to create one row
 		fitDataB.push_back(sslice());

		is.getline(input,200);
		string line = string(input);
		//cout << "line = " << line << endl;
		util_splitString(line, ',', fields);
		for (int j=0; j<6; ++j) {
			//cout << "fields[" << j << "] = " <<  fields[j] << endl;
			fitDataB[i].push_back(fields[j]);
			//cout << "fitDataB[" << i << "][" << j << "] = "<<  fitDataB[i][j] << endl;
		}
        fields.clear();
	}

	return 1;

}

//
// ------------------------------ readPix0Data ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Reads the pix0 data from the pix0 fit file
//
// inputs:
//  is  -  Reference to the input stream
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, July 2014
// =============================================================================
int DFMS_X0FitFile_Class::readFitResData(fstream &is) {

	string sFunctionName = "DFMS_X0FitFile_Class::readPix0Data";

	char input[200];
	vector<string> fields;
	string str;

	int numCols = 4;
	//int numCols = 6;

	// Now read the HK data into the HKdata string array
	for (int iRow=0; iRow<pixFitRows; iRow++) {
		//is.getline(input,100,',');
		is.getline(input,200);
		str.assign(input);
		//cout << "str = " << str << endl;
		//split data on "," and set to each array position
		util_splitString(str, ',', fields);
		int vSize = fields.size();
		for (int i=0; i<numCols; ++i) {
			//cout << "fields[" << i << "] = " <<  fields[i] << endl;
			fitResData[i] = (util_stringToDouble(fields[i]));
			//cout << "fitResData[" << i << "] = " <<  fitResData[i] << endl;
		}
		if (iRow == 0) {
			aA = fitResData[0];
			bA = fitResData[1];
			sErrorA = fitResData[2];
			R2coeffA = fitResData[3];
			fields.clear();
		}
		if (iRow == 1) {
			aB = fitResData[0];
			bB = fitResData[1];
			sErrorB = fitResData[2];
			R2coeffB = fitResData[3];
		}
	}

	return 1;
}

//
// -----------------------------  readPix0FitFile ------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Performs all the steps need to open/read/close a pix0 fit file and store the
// various data into arrays
//
// inputs:
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, July 2014
// =============================================================================
int DFMS_X0FitFile_Class::readPix0FitFile(int iRow, string type, string resType, string massType,
						string sRefDateAndTime, vector<fitFileInfo> &fitInfo ) {

     string sFunctionName = "DFMS_X0FitFile_Class::readPix0FitFile";

	fstream is;
	int nSuccess;
	string sRow;
	if (iRow == 1) sRow = "A";
	if (iRow == 2) sRow = "B";

	// Use the static method findBestFitFile to search all the available fit files
	// for the one closest in time
	nSuccess = findBestFitFile(sx0FitsDataPath, sRefDateAndTime, type, resType, massType, fitInfo);
	if (!nSuccess) {
		sErrorMessage = "Unable to find the best X0 PDS file ";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		//cout << "fileName = " << fileName << endl;
		return 0;
	} else {
		sInfoMessage = "The best fit file for Row "+sRow+ " is:"+sBestFitFile;
		writeToLog(sInfoMessage, sFunctionName, INFO);
		if (verbose >= 3) cout << sInfoMessage << endl;
		cout << sInfoMessage << endl;
	}

	// Open the L2 File stream //
	fileName = sx0FitsDataPath+sBestFitFile;
	nSuccess = openFile(fileName,is,1);
	if (!nSuccess) {
		sErrorMessage = "Unable to open X0Fitfile PDS file: ";
		sErrorMessage += fileName+" - Skipping file processing";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		if (is.is_open()) is.close();
		return 0;
	}

        // Read the pix0fit file PDS header
     nSuccess = readPDSheader(is);
     if (!nSuccess) {
          sErrorMessage = "Unable to read the PDS header for file: ";
          sErrorMessage += fileName+" - Skipping file processing";
          writeToLog(sErrorMessage, sFunctionName, ERROR);
          return 0;
     }
     // Get the Quality Id flag
     qualID = util_stringToInt(util_trimChar(PDSheader["DATA_QUALITY_ID"],"\""));
     //if (verbose == 1) printL2File("PDSHeader");

     // Read the pix0 Fit File fit A object description
     nSuccess = readObj(is, fitObjA, fitRowsA, fitCols);
     if (!nSuccess) {
    	 sErrorMessage = "Unable to read the Fit A Object Descp. for file: ";
    	 sErrorMessage += fileName+" - Skipping file processing";
    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
    	 return 0;
     }
     //if (verbose == 1) printL2File("CalObj");

     // Read the pix0 Fit File fit B object description
     nSuccess = readObj(is, fitObjB, fitRowsB, fitCols);
     if (!nSuccess) {
    	 sErrorMessage = "Unable to read the Fit B Object Descp. for file: ";
    	 sErrorMessage += fileName+" - Skipping file processing";
    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
    	 return 0;
     }
     //if (verbose == 1) printL2File("CalObj");

     // Read the pix0 Fit File Data object description
     nSuccess = readObj(is, fitResDataObj, pixFitRows, fitCols);
     if (!nSuccess) {
    	 sErrorMessage = "Unable to read the Data Object Descp. for file: ";
    	 sErrorMessage += fileName+" - Skipping file processing";
    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
    	 return 0;
     }
     //if (verbose == 1) printL2File("pix0DataObj");

     // Read the Fit Data
     nSuccess = readFitData(is);
     if (!nSuccess) {
    	 sErrorMessage = "Unable to read the Fit Data for file: ";
    	 sErrorMessage += fileName+" - Skipping file processing";
    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
    	 return 0;
     }

     //if (verbose == 1) printL2File("fitData");

     // Read the fit results Data
     nSuccess = readFitResData(is);
     if (!nSuccess) {
    	 sErrorMessage = "Unable to read the Fit results Data for file: ";
    	 sErrorMessage += fileName+" - Skipping file processing";
    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
    	 return 0;
     }
     //if (verbose == 1) printL2File("pix0Data");
     //cout << "fileName = " << sBestFitFile << endl;
     // Close the L2 File stream
     closeFile(is);

     return 1;
}

//
//  --------------------------- formFitObj ---------------------------------
//
int DFMS_X0FitFile_Class::formFitObj(int iRow, int fitRows) {

	if (iRow == 1) {
		fitObjA["OBJECT"]             = "DFMS_CALA_TABLE";
		fitObjA["NAME"]               = "DFMS_CALA_TABLE";
		fitObjA["INTERCHANGE_FORMAT"] = "ASCII";
		fitObjA["ROWS"] = util_intToString(fitRows);
		fitObjA["COLUMNS"]            = "7";
		fitObjA["ROW_BYTES"]          = "80";
		fitObjA["^STRUCTURE"]         = "\"DFMS_PIX0FIT_CAL_TABLE.FMT\"";
		fitObjA["END_OBJECT"]         = "DFMS_CALA_TABLE";
	} else {
		fitObjB["OBJECT"]             = "DFMS_CALB_TABLE";
		fitObjB["NAME"]               = "DFMS_CALB_TABLE";
		fitObjB["INTERCHANGE_FORMAT"] = "ASCII";
		fitObjB["ROWS"] = util_intToString(fitRows);
		fitObjB["COLUMNS"]            = "7";
		fitObjB["ROW_BYTES"]          = "80";
		fitObjB["^STRUCTURE"]         = "\"DFMS_PIX0FIT_CAL_TABLE.FMT\"";
		fitObjB["END_OBJECT"]         = "DFMS_CALB_TABLE";
	}
	return 1;
}

//
//  -------------------------- formFitResDataObj -----------------------------
//
int DFMS_X0FitFile_Class::formFitResDataObj() {

	fitResDataObj["OBJECT"]             = "DFMS_FITRESULTS_TABLE";
	fitResDataObj["NAME"]               = "DFMS_FITRESULTS_TABLE";
	fitResDataObj["INTERCHANGE_FORMAT"] = "ASCII";
	fitResDataObj["ROWS"]               = util_intToString(pixFitRows);
	fitResDataObj["COLUMNS"]            = "5";
	fitResDataObj["ROW_BYTES"]          = "80";
	fitResDataObj["^STRUCTURE"]         = "\"DFMS_PIX0FIT_COEFF_TABLE.FMT\"";
	fitResDataObj["END_OBJECT"]         = "DFMS_FITRESULTS_TABLE";

	return 1;
}

//
// ----------------------------- writePDSheader ------------------------------------
//
int DFMS_X0FitFile_Class::writePDSheader(fstream &os, string l2File) {

	string sFunctionName = "DFMS_X0FitFile_Class::writePDSheader";

    char tmp[810];
    string stmp;

	// get the L2 PDS header
    int nSuccess = getL2header(sL2sourcePath,l2File);
    if (!nSuccess) {
       	 sErrorMessage = "Unable to get L2 header for X0 Fit File: ";
       	 writeToLog(sErrorMessage, sFunctionName, ERROR);
       	 cout << sErrorMessage << endl;
       	 return 0;
    }

	int lblRec = 64;
	int pix0recs = 2;
	int lastLineNum = lblRec+fitRowsA+fitRowsB+pix0recs;
    stmp = "PDS_VERSION_ID                   =     ";
    	L2header["PDS_VERSION_ID"] = util_trimEOL(L2header["PDS_VERSION_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["PDS_VERSION_ID"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "LABEL_REVISION_NOTE              =     \"2015-01-01,Thierry Semon(UoB),";
    	sprintf(tmp,"%-78s",stmp.c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "                                        Version1.0 release\"";
    	sprintf(tmp,"%-78s",stmp.c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "RECORD_TYPE                      =     ";
    	L2header["RECORD_TYPE"] = util_trimEOL(L2header["RECORD_TYPE"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["RECORD_TYPE"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "RECORD_BYTES                     =     ";
    	L2header["RECORD_BYTES"] = util_trimEOL(L2header["RECORD_BYTES"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["RECORD_BYTES"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "FILE_RECORDS                     =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lastLineNum)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "LABEL_RECORDS                    =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lblRec)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "^DFMS_CALA_TABLE                 =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lblRec+1)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "^DFMS_CALB_TABLE                 =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lblRec+fitRowsA+1)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "^DFMS_FITRESULTS_TABLE           =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lblRec+fitRowsA+fitRowsB+1)).c_str());
       os << string(tmp) << dfmsEOL;

    string newDName = L2header["DATA_SET_ID"];
    replace(newDName,"-2-","-3-");
    stmp = "DATA_SET_ID                      =     ";
    	newDName=util_trimEOL(newDName);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),newDName.c_str());
    	os << string(tmp) << dfmsEOL;

    newDName = L2header["DATA_SET_NAME"];
    replace(newDName,"2","3");
    stmp = "DATA_SET_NAME                    =     ";
		newDName=util_trimEOL(newDName);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),newDName.c_str());
		os << string(tmp) << dfmsEOL;

       // Modify the PDS Header - PRODUCT_ID
       // X0_GCU_20140720_083106_HMLR.TAB
	string name = fileName.substr(fileName.length()-31,27);
    stmp = "PRODUCT_ID                       =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),name.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "PRODUCT_CREATION_TIME            =     ";
       string tTime = getProcessTime(1);
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),tTime.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "PRODUCT_TYPE                     =     RDR";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "PROCESSING_LEVEL_ID              =     \"3\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "MISSION_ID                       =     ";
    	L2header["MISSION_ID"] = util_trimEOL(L2header["MISSION_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["MISSION_ID"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "MISSION_NAME                     =     ";
    	L2header["MISSION_NAME"] = util_trimEOL(L2header["MISSION_NAME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["MISSION_NAME"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_HOST_NAME             =     ";
    	L2header["INSTRUMENT_HOST_NAME"] = util_trimEOL(L2header["INSTRUMENT_HOST_NAME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["INSTRUMENT_HOST_NAME"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_HOST_ID               =     ";
    	L2header["INSTRUMENT_HOST_ID"] = util_trimEOL(L2header["INSTRUMENT_HOST_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["INSTRUMENT_HOST_ID"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_NAME                  =     ";
    	L2header["INSTRUMENT_NAME"] = util_trimEOL(L2header["INSTRUMENT_NAME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["INSTRUMENT_NAME"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_ID                    =     ";
    	L2header["INSTRUMENT_ID"] = util_trimEOL(L2header["INSTRUMENT_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["INSTRUMENT_ID"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_TYPE                  =     ";
    	L2header["INSTRUMENT_TYPE"] = util_trimEOL(L2header["INSTRUMENT_TYPE"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["INSTRUMENT_TYPE"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "DETECTOR_ID                      =     ";
    	L2header["DETECTOR_ID"] = util_trimEOL(L2header["DETECTOR_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["DETECTOR_ID"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "DETECTOR_DESC                    =     ";
    	L2header["DETECTOR_DESC"] = util_trimEOL(L2header["DETECTOR_DESC"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["DETECTOR_DESC"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "PRODUCER_ID                      =     ";
    	L2header["PRODUCER_ID"] = util_trimEOL(L2header["PRODUCER_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["PRODUCER_ID"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "PRODUCER_FULL_NAME               =     ";
    	L2header["PRODUCER_FULL_NAME"] = util_trimEOL(L2header["PRODUCER_FULL_NAME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["PRODUCER_FULL_NAME"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "PRODUCER_INSTITUTION_NAME        =     ";
    	L2header["PRODUCER_INSTITUTION_NAME"] = util_trimEOL(L2header["PRODUCER_INSTITUTION_NAME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),L2header["PRODUCER_INSTITUTION_NAME"].c_str());
    	os << string(tmp) << dfmsEOL;

    string qid = "\""+util_intToString(qualID)+"\"";
    stmp = "DATA_QUALITY_ID                  =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),qid.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "DATA_QUALITY_DESC                =     \"";
    	sprintf(tmp,"%-78s",stmp.c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "      0 means 'Nominal quality, Fit was successful\'";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "      1 means 'Less than 4 points used\'";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "      2 means 'R^2 is low for more than 4 points\'";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "      3 means 'Peak is out of pixel range 200-310\'\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;
    // Add an 79 character blank line to separate PDSheader from first Object descriptor

       sprintf(tmp,"%-78s"," ");
	   os << string(tmp) << dfmsEOL;

	// Clear the header for next time
	L2header.clear();

	return 1;

}


//
//  -------------------------- writeFitData -------------------------------
//
int DFMS_X0FitFile_Class::writeFitData(fstream &os, string type, string massType, string resType) {

	string sFunctionName = "DFMS_X0FitFile_Class::writeFitData";

	char tmp[100];
	char line[200];
	string sOut;
	string blank = " ";
	string s;

	for (int i=0; i<fitRowsA; i++) {
		// Get data into proper form
		if (type.compare("GCU") == 0) {
			if (resType.compare("L") == 0) {
				if (massType.compare("L") == 0) {
					fitDataA[i][0] = (speciesDef[gcuLmLrA[i].m0]).name;
					sprintf(tmp,"%-6.2f",gcuLmLrA[i].m0);
					fitDataA[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",gcuLmLrA[i].m);
					fitDataA[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuLmLrA[i].pix0);
					fitDataA[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuLmLrA[i].pix0Unc);
					fitDataA[i][4] = s.assign(tmp);
					fitDataA[i][5] = gcuLmLrA[i].file.substr(0,27);
				} else if (massType.compare("H") == 0) {
					fitDataA[i][0] = (speciesDef[gcuHmLrA[i].m0]).name;
					sprintf(tmp,"%-6.2f",gcuHmLrA[i].m0);
					fitDataA[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",gcuHmLrA[i].m);
					fitDataA[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuHmLrA[i].pix0);
					fitDataA[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuHmLrA[i].pix0Unc);
					fitDataA[i][4] = s.assign(tmp);
					fitDataA[i][5] = gcuHmLrA[i].file.substr(0,27);
				}
			} else if (resType.compare("H") == 0) {
				if (massType.compare("L") == 0) {
					fitDataA[i][0] = (speciesDef[gcuLmHrA[i].m0]).name;
					sprintf(tmp,"%-6.2f",gcuLmHrA[i].m0);
					fitDataA[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",gcuLmHrA[i].m);
					fitDataA[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuLmHrA[i].pix0);
					fitDataA[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuLmHrA[i].pix0Unc);
					fitDataA[i][4] = s.assign(tmp);
					fitDataA[i][5] = gcuLmHrA[i].file.substr(0,27);
				} else if (massType.compare("M") == 0) {
					fitDataA[i][0] = (speciesDef[gcuMmHrA[i].m0]).name;
					sprintf(tmp,"%-6.2f",gcuMmHrA[i].m0);
					fitDataA[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",gcuMmHrA[i].m);
					fitDataA[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuMmHrA[i].pix0);
					fitDataA[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuMmHrA[i].pix0Unc);
					fitDataA[i][4] = s.assign(tmp);
					fitDataA[i][5] = gcuMmHrA[i].file.substr(0,27);
				} else if (massType.compare("H") == 0) {
					fitDataA[i][0] = (speciesDef[gcuHmHrA[i].m0]).name;
					sprintf(tmp,"%-6.2f",gcuHmHrA[i].m0);
					fitDataA[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",gcuHmHrA[i].m);
					fitDataA[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuHmHrA[i].pix0);
					fitDataA[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuHmHrA[i].pix0Unc);
					fitDataA[i][4] = s.assign(tmp);
					fitDataA[i][5] = gcuHmHrA[i].file.substr(0,27);
				}
			}
		}
		if (type.compare("SLF") == 0) {
			if (resType.compare("L") == 0) {
				if (massType.compare("L") == 0) {
					fitDataA[i][0] = (speciesDef[slfLmLrA[i].m0]).name;
					sprintf(tmp,"%-6.2f",slfLmLrA[i].m0);
					fitDataA[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",slfLmLrA[i].m);
					fitDataA[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfLmLrA[i].pix0);
					fitDataA[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfLmLrA[i].pix0Unc);
					fitDataA[i][4] = s.assign(tmp);
					fitDataA[i][5] = slfLmLrA[i].file.substr(0,27);
				} else if (massType.compare("H") == 0) {
					fitDataA[i][0] = (speciesDef[slfHmLrA[i].m0]).name;
					sprintf(tmp,"%-6.2f",slfHmLrA[i].m0);
					fitDataA[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",slfHmLrA[i].m);
					fitDataA[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfHmLrA[i].pix0);
					fitDataA[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfHmLrA[i].pix0Unc);
					fitDataA[i][4] = s.assign(tmp);
					fitDataA[i][5] = slfHmLrA[i].file.substr(0,27);
				}
			} else if (resType.compare("H") == 0) {
				if (massType.compare("L") == 0) {
					fitDataA[i][0] = (speciesDef[slfLmHrA[i].m0]).name;
					sprintf(tmp,"%-6.2f",slfLmHrA[i].m0);
					fitDataA[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",slfLmHrA[i].m);
					fitDataA[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfLmHrA[i].pix0);
					fitDataA[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfLmHrA[i].pix0Unc);
					fitDataA[i][4] = s.assign(tmp);
					fitDataA[i][5] = slfLmHrA[i].file.substr(0,27);
				} else if (massType.compare("M") == 0) {
					fitDataA[i][0] = (speciesDef[slfMmHrA[i].m0]).name;
					sprintf(tmp,"%-6.2f",slfMmHrA[i].m0);
					fitDataA[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",slfMmHrA[i].m);
					fitDataA[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfMmHrA[i].pix0);
					fitDataA[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfMmHrA[i].pix0Unc);
					fitDataA[i][4] = s.assign(tmp);
					fitDataA[i][5] = slfMmHrA[i].file.substr(0,27);
				} else if (massType.compare("H") == 0) {
					fitDataA[i][0] = (speciesDef[slfHmHrA[i].m0]).name;
					sprintf(tmp,"%-6.2f",slfHmHrA[i].m0);
					fitDataA[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",slfHmHrA[i].m);
					fitDataA[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfHmHrA[i].pix0);
					fitDataA[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfHmHrA[i].pix0Unc);
					fitDataA[i][4] = s.assign(tmp);
					fitDataA[i][5] = slfHmHrA[i].file.substr(0,27);
				}
			}
		}
		sprintf(line,"\"%-12.12s\", %5.5s,%10.10s,%7.7s, %4.4s,\"%27.27s\", ",
		(fitDataA[i][0]).c_str(), (fitDataA[i][1]).c_str(), (fitDataA[i][2]).c_str(),
		(fitDataA[i][3]).c_str(), (fitDataA[i][4]).c_str(), (fitDataA[i][5]).c_str());

		sOut = string(line);
		os << sOut << dfmsEOL;
	}

	sOut.clear();

	for (int i=0; i<fitRowsB; i++) {
		// Get data into proper form
		if (type.compare("GCU") == 0) {
			if (resType.compare("L") == 0) {
				if (massType.compare("L") == 0) {
					fitDataB[i][0] = (speciesDef[gcuLmLrB[i].m0]).name;
					sprintf(tmp,"%-6.2f",gcuLmLrB[i].m0);
					fitDataB[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",gcuLmLrB[i].m);
					fitDataB[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuLmLrB[i].pix0);
					fitDataB[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuLmLrB[i].pix0Unc);
					fitDataB[i][4] = s.assign(tmp);
					fitDataB[i][5] = gcuLmLrB[i].file.substr(0,27);
				} else if (massType.compare("H") == 0) {
					fitDataB[i][0] = (speciesDef[gcuHmLrB[i].m0]).name;
					sprintf(tmp,"%-6.2f",gcuHmLrB[i].m0);
					fitDataB[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",gcuHmLrB[i].m);
					fitDataB[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuHmLrB[i].pix0);
					fitDataB[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuHmLrB[i].pix0Unc);
					fitDataB[i][4] = s.assign(tmp);
					fitDataB[i][5] = gcuHmLrB[i].file.substr(0,27);
				}
			} else if (resType.compare("H") == 0) {
				if (massType.compare("L") == 0) {
					fitDataB[i][0] = (speciesDef[gcuLmHrB[i].m0]).name;
					sprintf(tmp,"%-6.2f",gcuLmHrB[i].m0);
					fitDataB[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",gcuLmHrB[i].m);
					fitDataB[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuLmHrB[i].pix0);
					fitDataB[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuLmHrB[i].pix0Unc);
					fitDataB[i][4] = s.assign(tmp);
					fitDataB[i][5] = gcuLmHrB[i].file.substr(0,27);
				} else if (massType.compare("M") == 0) {
					fitDataB[i][0] = (speciesDef[gcuMmHrB[i].m0]).name;
					sprintf(tmp,"%-6.2f",gcuMmHrB[i].m0);
					fitDataB[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",gcuMmHrB[i].m);
					fitDataB[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuMmHrB[i].pix0);
					fitDataB[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuMmHrB[i].pix0Unc);
					fitDataB[i][4] = s.assign(tmp);
					fitDataB[i][5] = gcuMmHrB[i].file.substr(0,27);
				} else if (massType.compare("H") == 0) {
					fitDataB[i][0] = (speciesDef[gcuHmHrB[i].m0]).name;
					sprintf(tmp,"%-6.2f",gcuHmHrB[i].m0);
					fitDataB[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",gcuHmHrB[i].m);
					fitDataB[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuHmHrB[i].pix0);
					fitDataB[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",gcuHmHrB[i].pix0Unc);
					fitDataB[i][4] = s.assign(tmp);
					fitDataB[i][5] = gcuHmHrB[i].file.substr(0,27);
				}
			}
		}
		if (type.compare("SLF") == 0) {
			if (resType.compare("L") == 0) {
				if (massType.compare("L") == 0) {
					fitDataB[i][0] = (speciesDef[slfLmLrB[i].m0]).name;
					sprintf(tmp,"%-6.2f",slfLmLrB[i].m0);
					fitDataB[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",slfLmLrB[i].m);
					fitDataB[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfLmLrB[i].pix0);
					fitDataB[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfLmLrB[i].pix0Unc);
					fitDataB[i][4] = s.assign(tmp);
					fitDataB[i][5] = slfLmLrB[i].file.substr(0,27);
				} else if (massType.compare("H") == 0) {
					fitDataB[i][0] = (speciesDef[slfHmLrB[i].m0]).name;
					sprintf(tmp,"%-6.2f",slfHmLrB[i].m0);
					fitDataB[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",slfHmLrB[i].m);
					fitDataB[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfHmLrB[i].pix0);
					fitDataB[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfHmLrB[i].pix0Unc);
					fitDataB[i][4] = s.assign(tmp);
					fitDataB[i][5] = slfHmLrB[i].file.substr(0,27);
				}
			} else if (resType.compare("H") == 0) {
				if (massType.compare("L") == 0) {
					fitDataB[i][0] = (speciesDef[slfLmHrB[i].m0]).name;
					sprintf(tmp,"%-6.2f",slfLmHrB[i].m0);
					fitDataB[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",slfLmHrB[i].m);
					fitDataB[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfLmHrB[i].pix0);
					fitDataB[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfLmHrB[i].pix0Unc);
					fitDataB[i][4] = s.assign(tmp);
					fitDataB[i][5] = slfLmHrB[i].file.substr(0,27);
				} else if (massType.compare("M") == 0) {
					fitDataB[i][0] = (speciesDef[slfMmHrB[i].m0]).name;
					sprintf(tmp,"%-6.2f",slfMmHrB[i].m0);
					fitDataB[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",slfMmHrB[i].m);
					fitDataB[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfMmHrB[i].pix0);
					fitDataB[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfMmHrB[i].pix0Unc);
					fitDataB[i][4] = s.assign(tmp);
					fitDataB[i][5] = slfMmHrB[i].file.substr(0,27);
				} else if (massType.compare("H") == 0) {
					fitDataB[i][0] = (speciesDef[slfHmHrB[i].m0]).name;
					sprintf(tmp,"%-6.2f",slfHmHrB[i].m0);
					fitDataB[i][1] = s.assign(tmp);
					sprintf(tmp,"%10.6f",slfHmHrB[i].m);
					fitDataB[i][2] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfHmHrB[i].pix0);
					fitDataB[i][3] = s.assign(tmp);
					sprintf(tmp,"%-6.2f",slfHmHrB[i].pix0Unc);
					fitDataB[i][4] = s.assign(tmp);
					fitDataB[i][5] = slfHmHrB[i].file.substr(0,27);
				}
			}
		}
		sprintf(line,"\"%-12.12s\", %5.5s,%10.10s,%7.7s, %4.4s,\"%27.27s\", ",
		(fitDataB[i][0]).c_str(), (fitDataB[i][1]).c_str(), (fitDataB[i][2]).c_str(),
		(fitDataB[i][3]).c_str(), (fitDataB[i][4]).c_str(), (fitDataB[i][5]).c_str());

		sOut = string(line);
		cout << sOut << endl;
		os << sOut << dfmsEOL;
	}

	return 1;
}

//
//  -------------------------- writePix0Data ------------------------------
//
int DFMS_X0FitFile_Class::writeFitResData(fstream &os) {

	string sFunctionName = "DFMS_PDS_L3_file_Class::writeL3data";

	char line1[100];
	char line2[100];
	string sOut;

	// Write Row A Fit Results
	//sprintf(line1,"%10.4e, %10.4e, %10.4e, %10.4e, %10.4e, %10.4e, %7s",
	//		       aA, errOnaA, bA, errOnbA, sErrorA, R2coeffA, " ");
	sprintf(line1,"%10.4e, %11.4e, %10.4e, %10.4e,%30.30s", aA, bA, sErrorA, R2coeffA,"");
	sOut = string(line1);
	os << sOut << dfmsEOL;

	// Write Row B Fit Results
	//sprintf(line2,"%10.4e, %10.4e, %10.4e, %10.4e, %10.4e, %10.4e, %7s",
	//	       	   aB, errOnaB, bB, errOnbB, sErrorB, R2coeffB, " ");
	sprintf(line2,"%10.4e, %11.4e, %10.4e, %10.4e,%30.30s", aB, bB, sErrorB, R2coeffB,"");
	sOut = string(line2);
	os << sOut << dfmsEOL;

	return 1;
}

//
//  -------------------------- makePix0FitFile -----------------------------
//
int DFMS_X0FitFile_Class::writePix0FitFile(string type, string massType, string resType, string l2File) {

	string sFunctionName = "DFMS_X0FitFile_Class::writePix0FitFile";

	int rows,cols;
	fstream os;
	int nSuccess;

	// Open the L3 File stream for output
	openFile(fileName,os,0);

	// Form the Fit row A data descp obj
	nSuccess = formFitObj(1, fitRowsA);
	if (!nSuccess) {
		sErrorMessage = "Unable to form the Fit Row A data Object \n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Form the Fit row B data descp obj
	nSuccess = formFitObj(2, fitRowsB);
	if (!nSuccess) {
		sErrorMessage = "Unable to form the Fit Row B data Object \n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Form the Fit Results data descp obj
	nSuccess = formFitResDataObj();
	if (!nSuccess) {
		sErrorMessage = "Unable to form the Fit Results data Object\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// write the L3 file PDS header
	nSuccess = writePDSheader(os, l2File);
	if (!nSuccess) {
		sErrorMessage = "Unable to write the PDS header for file: ";
		sErrorMessage += fileName;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printPix0File("PDSHeader");

	// Write the fit file row A object description
	nSuccess = writeObj(os, fitObjA, 0);
	if (!nSuccess) {
		sErrorMessage = "Unable to write Fit file Row A Descrp. Object for file: ";
		sErrorMessage += fileName;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printPix0File("fitObjA");

	// Write the fit file row B object description
	nSuccess = writeObj(os, fitObjB, 0);
	if (!nSuccess) {
		sErrorMessage = "Unable to write Fit file Row B Descrp. Object for file: ";
		sErrorMessage += fileName;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printPix0File("fitObjB");

	// Write the pix0 Fit Results Data object description
	nSuccess = writeObj(os, fitResDataObj, 1);
	if (!nSuccess) {
		sErrorMessage = "Unable to write the pix0 Fit Results data Object Descp. for file: ";
		sErrorMessage += fileName;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printPix0File("pix0DataObj");

	// Write the fit file Data
	nSuccess = writeFitData(os, type, massType, resType);
	if (!nSuccess) {
		sErrorMessage = "Unable to write the Fit Data to file: ";
		sErrorMessage += fileName;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printPix0File("fitData");

	// Write the Fit Results to file L3 Data
	nSuccess = writeFitResData(os);
	if (!nSuccess) {
		sErrorMessage = "Unable to write the Fit Results Data to file: ";
		sErrorMessage += fileName;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printPix0File("fitResData");

	// Close the L3 File stream
	closeFile(os);

	return 1;

}

//
// ---------------------------- doTheLinFit ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method does a linear fit of 'type' mass vs 'type' pix0, where 'type' is
// one of 'gcu' or 'slf'
//
// inputs:
//   type  -  gcu or slf
//   massType - Low or High mass
//   resType - Low or High resolution data
//   gcuP -
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, June 2014
// =============================================================================
//
int DFMS_X0FitFile_Class::doTheLinFit(string type, string massType, string resType,
									   linFitParams &gcuP) {

	int numM0=0;
	DFMS_LinFit_Class *linFitA;
	DFMS_LinFit_Class *linFitB;
	double singleM0 = 0.0;
	int numUmass = 0;

	if (type.compare("GCU") == 0) {
		// When we have at least MIN4GCUFIT new GCU files then do a fit and write
		// parameters to file.  Write new file/parameters as more GCU files
		// create more points for better fit. This guarantee's we have a
		// file to read with at least a MIN4GCUFIT pt fit.
		if (resType.compare("L") == 0) {
			if (massType.compare("L") == 0) {

				linFitA = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
				if (numUmass > 3) {  // Check to see if we have more than 1 m0
					linFitA->startFit((long)gcuLmLrA.size(), "GCU", "L", "L", 1);
					aA = linFitA->geta();
					bA = linFitA->getb();
					RcoeffA = linFitA->getCoefC();
					R2coeffA = linFitA->getCoefD();
					//sErrorA = linFitA->getSerror();
					linFitA->errors("GCU", "L", "L", 1);
					sErrorA = linFitA->getSigEst();
					errOnaA = linFitA->getSa();
					errOnbA = linFitA->getSb();
				} else {
					bA = gcuP.bA;
					double pix0 = linFitA->avgPix0("GCU", "L", "L", 1);
					aA = determineA(pix0, bA, singleM0);
					RcoeffA = 0.0;
					R2coeffA = 0.0;
					sErrorA = 0.0;
					errOnaA = 0.0;
					errOnbA = 0.0;
				}

				linFitB = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
				if (numUmass > 3) {
					linFitB->startFit((long)gcuLmLrB.size(), "GCU", "L", "L", 2);
					aB = linFitB->geta();
					bB = linFitB->getb();
					RcoeffB = linFitB->getCoefC();
					R2coeffB = linFitB->getCoefD();
					//sErrorB = linFitB->getSerror();
					linFitB->errors("GCU", "L", "L", 2);
					sErrorB = linFitB->getSigEst();
					errOnaB = linFitB->getSa();
					errOnbB = linFitB->getSb();
				} else {
					bB = gcuP.bB;
					double pix0 = linFitB->avgPix0("GCU", "L", "L", 2);
					aB = determineA(pix0, bB, singleM0);
					RcoeffB = 0.0;
					R2coeffB = 0.0;
					sErrorB = 0.0;
					errOnaB = 0.0;
					errOnbB = 0.0;
				}

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;
			}
			// Write High Mass fit
			if (massType.compare("H") == 0) {

				linFitA = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
				if (numUmass > 3) {
					linFitA->startFit((long)gcuHmLrA.size(), "GCU", "L", "H", 1);
					aA = linFitA->geta();
					bA = linFitA->getb();
					RcoeffA = linFitA->getCoefC();
					R2coeffA = linFitA->getCoefD();
					//sErrorA = linFitA->getSerror();
					linFitA->errors("GCU", "L", "H", 1);
					sErrorA = linFitA->getSigEst();
					errOnaA = linFitA->getSa();
					errOnbA = linFitA->getSb();
				} else {
					bA = gcuP.bA;
					double pix0 = linFitA->avgPix0("GCU", "L", "H", 1);
					aA = determineA(pix0, bA, singleM0);
					RcoeffA = 0.0;
					R2coeffA = 0.0;
					sErrorA = 0.0;
					errOnaA = 0.0;
					errOnbA = 0.0;
				}

				linFitB = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
				if (numUmass > 3) {
					linFitB->startFit((long)gcuHmLrB.size(), "GCU", "L", "H", 2);
					aB = linFitB->geta();
					bB = linFitB->getb();
					RcoeffB = linFitB->getCoefC();
					R2coeffB = linFitB->getCoefD();
					//sErrorB = linFitB->getSerror();
					linFitB->errors("GCU", "L", "H", 2);
					sErrorB = linFitB->getSigEst();
					errOnaB = linFitB->getSa();
					errOnbB = linFitB->getSb();
				} else {
					bB = gcuP.bB;
					double pix0 = linFitB->avgPix0("GCU", "L", "H", 2);
					aB = determineA(pix0, bB, singleM0);
					RcoeffB = 0.0;
					R2coeffB = 0.0;
					sErrorB = 0.0;
					errOnaB = 0.0;
					errOnbB = 0.0;
				}

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;

			}
		} else if (resType.compare("H") == 0) {

			if (massType.compare("L") == 0) {

				linFitA = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
				if (numUmass > 2) {
					// Check to see if we have more than 1 m0
					linFitA->startFit((long)gcuLmHrA.size(), "GCU", "H", "L", 1);
					aA = linFitA->geta();
					bA = linFitA->getb();
					RcoeffA = linFitA->getCoefC();
					R2coeffA = linFitA->getCoefD();
					//sErrorA = linFitA->getSerror();
					linFitA->errors("GCU", "H", "L", 1);
					sErrorA = linFitA->getSigEst();
					errOnaA = linFitA->getSa();
					errOnbA = linFitA->getSb();
				} else {
					bA = gcuP.bA;
					double pix0 = linFitA->avgPix0("GCU", "H", "L", 1);
					aA = determineA(pix0, bA, singleM0);
					RcoeffA = 0.0;
					R2coeffA = 0.0;
					sErrorA = 0.0;
					errOnaA = 0.0;
					errOnbA = 0.0;
				}

				linFitB = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
				if (numUmass > 2) {
					linFitB->startFit((long)gcuLmHrB.size(), "GCU", "H", "L", 2);
					aB = linFitB->geta();
					bB = linFitB->getb();
					RcoeffB = linFitB->getCoefC();
					R2coeffB = linFitB->getCoefD();
					//sErrorB = linFitB->getSerror();
					linFitB->errors("GCU", "H", "L", 2);
					sErrorB = linFitB->getSigEst();
					errOnaB = linFitB->getSa();
					errOnbB = linFitB->getSb();
				} else {
					bB = gcuP.bB;
					double pix0 = linFitB->avgPix0("GCU", "H", "L", 2);
					aB = determineA(pix0, bB, singleM0);
					RcoeffB = 0.0;
					R2coeffB = 0.0;
					sErrorB = 0.0;
					errOnaB = 0.0;
					errOnbB = 0.0;
				}

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;
			}

			// Write Medium Mass fit
			if (massType.compare("M") == 0) {

				linFitA = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
				if (numUmass > 2) {
					// Check to see if we have more than 1 m0
					linFitA->startFit((long)gcuMmHrA.size(), "GCU", "H", "M", 1);
					aA = linFitA->geta();
					bA = linFitA->getb();
					RcoeffA = linFitA->getCoefC();
					R2coeffA = linFitA->getCoefD();
					linFitA->errors("GCU", "H", "M", 1);
					sErrorA = linFitA->getSigEst();
					errOnaA = linFitA->getSa();
					errOnbA = linFitA->getSb();
				} else {
					bA = gcuP.bA;
					double pix0 = linFitA->avgPix0("GCU", "H", "M", 1);
					aA = determineA(pix0, bA, singleM0);
					RcoeffA = 0.0;
					R2coeffA = 0.0;
					sErrorA = 0.0;
					errOnaA = 0.0;
					errOnbA = 0.0;
				}

				linFitB = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
				if (numUmass > 2) {
					linFitB->startFit((long)gcuMmHrB.size(), "GCU", "H", "M", 2);
					aB = linFitB->geta();
					bB = linFitB->getb();
					RcoeffB = linFitB->getCoefC();
					R2coeffB = linFitB->getCoefD();
					linFitB->errors("GCU", "H", "M", 2);
					sErrorB = linFitB->getSigEst();
					errOnaB = linFitB->getSa();
					errOnbB = linFitB->getSb();
				} else {
					bB = gcuP.bB;
					double pix0 = linFitB->avgPix0("GCU", "H", "M", 2);
					aB = determineA(pix0, bB, singleM0);
					RcoeffB = 0.0;
					R2coeffB = 0.0;
					sErrorB = 0.0;
					errOnaB = 0.0;
					errOnbB = 0.0;
				}

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;
			}

			// Write High Mass fit
			if (massType.compare("H") == 0) {

				linFitA = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
				if (numUmass > 2) {
					linFitA->startFit((long)gcuHmHrA.size(), "GCU", "H", "H", 1);
					aA = linFitA->geta();
					bA = linFitA->getb();
					RcoeffA = linFitA->getCoefC();
					R2coeffA = linFitA->getCoefD();
					//sErrorA = linFitA->getSerror();
					linFitA->errors("GCU", "H", "H", 1);
					sErrorA = linFitA->getSigEst();
					errOnaA = linFitA->getSa();
					errOnbA = linFitA->getSb();
				} else {
					bA = gcuP.bA;
					double pix0 = linFitA->avgPix0("GCU", "H", "H", 1);
					aA = determineA(pix0, bA, singleM0);
					RcoeffA = 0.0;
					R2coeffA = 0.0;
					sErrorA = 0.0;
					errOnaA = 0.0;
					errOnbA = 0.0;
				}

				linFitB = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
				if (numUmass > 2) {
					linFitB->startFit((long)gcuHmHrB.size(), "GCU", "H", "H", 2);
					aB = linFitB->geta();
					bB = linFitB->getb();
					RcoeffB = linFitB->getCoefC();
					R2coeffB = linFitB->getCoefD();
					//sErrorB = linFitB->getSerror();
					linFitB->errors("GCU", "H", "H", 2);
					sErrorB = linFitB->getSigEst();
					errOnaB = linFitB->getSa();
					errOnbB = linFitB->getSb();
				} else {
					bB = gcuP.bB;
					double pix0 = linFitB->avgPix0("GCU", "H", "H", 2);
					aB = determineA(pix0, bB, singleM0);
					RcoeffB = 0.0;
					R2coeffB = 0.0;
					sErrorB = 0.0;
					errOnaB = 0.0;
					errOnbB = 0.0;
				}

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;

			}
		}

		// Set Fit Quality ID flag
		if (fitRowsA > MIN4GCUFIT && R2coeffA < 0.90) {
			qualID = 2;
		} else if (fitRowsB > MIN4GCUFIT && R2coeffB < 0.90) {
			qualID = 2;
		} else {
			qualID = 0;
		}

		// If this SLF L2 file date is before Jan 3, 2015 then here we are not doing a
		// linear fit but instead are using the gcu value for the slope (b). We then reverse the
		// y intercept (a) calculation (a = pix0 - b*m0) for each m0,pix0 pair.
		// Then we take the average of all the a's as the final y-intercept.
	} else if (type.compare("SLF") == 0 && isRecentGCUavailable) {

		if (resType.compare("L") == 0) {
			if (massType.compare("L") == 0) {

				linFitA = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
				if (numUmass > 3) {
					// Check to see if we have more than 1 m0
					bA = gcuP.bA;
					aA = linFitA->findOffSet((long)slfLmLrA.size(), bA, "SLF", "L", "L", 1);
				} else {
					bA = gcuP.bA;
					double pix0 = linFitA->avgPix0("SLF", "L", "L", 1);
					aA = determineA(pix0, bA, singleM0);
				}

				linFitB = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
				if (numUmass > 3) {
					bB = gcuP.bB;
					aB = linFitB->findOffSet((long)slfLmLrB.size(), bB, "SLF", "L", "L", 2);
				} else {
					bB = gcuP.bB;
					double pix0 = linFitB->avgPix0("SLF", "L", "L", 2);
					aB = determineA(pix0, bB, singleM0);
				}

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;
			}
			// Write High Mass fit
			if (massType.compare("H") == 0) {

				linFitA = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
				if (numUmass > 2) {
					bA = gcuP.bA;
					aA = linFitA->findOffSet((long)slfHmLrA.size(), bA, "SLF", "L", "H", 1);
				} else {
					bA = gcuP.bA;
					double pix0 = linFitA->avgPix0("SLF", "L", "H", 1);
					aA = determineA(pix0, bA, singleM0);
				}

				linFitB = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
				if (numUmass > 2) {
					bB = gcuP.bB;
					aB = linFitB->findOffSet((long)slfHmLrB.size(), bB, "SLF", "L", "H", 2);
				} else {
					bB = gcuP.bB;
					double pix0 = linFitB->avgPix0("SLF", "L", "H", 2);
					aB = determineA(pix0, bB, singleM0);
				}

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;

			}

		} else if (resType.compare("H") == 0) {

			if (massType.compare("L") == 0) {

				linFitA = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
				if (numUmass > 2) {
					// Check to see if we have more than 1 m0
					bA = gcuP.bA;
					aA = linFitA->findOffSet((long)slfLmHrA.size(), bA, "SLF", "H", "L", 1);
				} else {
					bA = gcuP.bA;
					double pix0 = linFitA->avgPix0("SLF", "H", "L", 1);
					aA = determineA(pix0, bA, singleM0);
				}

				linFitB = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
				if (numUmass > 2) {
					bB = gcuP.bB;
					aB = linFitB->findOffSet((long)slfLmHrB.size(), bB, "SLF", "H", "L", 2);
				} else {
					bB = gcuP.bB;
					double pix0 = linFitB->avgPix0("SLF", "H", "L", 2);
					aB = determineA(pix0, bB, singleM0);
				}

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;
			}
			// Write Medium Mass fit
			if (massType.compare("M") == 0) {

				linFitA = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
				if (numUmass > 1) {
					// Check to see if we have more than 1 m0
					bA = gcuP.bA;
					aA = linFitA->findOffSet((long)slfMmHrA.size(), bA, "SLF", "H", "M", 1);
				} else {
					bA = gcuP.bA;
					double pix0 = linFitA->avgPix0("SLF", "H", "M", 1);
					aA = determineA(pix0, bA, singleM0);
				}

				linFitB = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
				if (numUmass > 1) {
					bB = gcuP.bB;
					aB = linFitB->findOffSet((long)slfMmHrB.size(), bB, "SLF", "H", "M", 2);
				} else {
					bB = gcuP.bB;
					double pix0 = linFitB->avgPix0("SLF", "H", "M", 2);
					aB = determineA(pix0, bB, singleM0);
				}

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;
			}
			// Write High Mass fit
			if (massType.compare("H") == 0) {

				linFitA = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
				if (numUmass > 2) {
					bA = gcuP.bA;
					aA = linFitA->findOffSet((long)slfHmHrA.size(), bA, "SLF", "H", "H", 1);
				} else {
					bA = gcuP.bA;
					double pix0 = linFitA->avgPix0("SLF", "H", "H", 1);
					aA = determineA(pix0, bA, singleM0);
				}

				linFitB = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
				if (numUmass > 2) {
					bB = gcuP.bB;
					aB = linFitB->findOffSet((long)slfHmHrB.size(), bB, "SLF", "H", "H", 2);
				} else {
					bB = gcuP.bB;
					double pix0 = linFitB->avgPix0("SLF", "H", "H", 2);
					aB = determineA(pix0, bB, singleM0);
				}

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;

			}
		}

		// For SLF files where GCU is available no fit is taking place therefore these are not meaningful.
		RcoeffA = 0.0;
		R2coeffA = 0.0;
		sErrorA = 0.0;
		errOnaA = 0.0;
		errOnbA = 0.0;
		RcoeffB = 0.0;
		R2coeffB = 0.0;
		sErrorB = 0.0;
		errOnaB = 0.0;
		errOnbB = 0.0;

		// Set Fit Quality ID flag
		if (fitRowsA > MIN4SLFFIT && R2coeffA < 0.90) {
			qualID = 2;
		} else if (fitRowsB > MIN4SLFFIT && R2coeffB < 0.90) {
			qualID = 2;
		} else {
			qualID = 0;
		}
	//
	// If this L2 file date is after Jan 3, 2015 then do a linear fit (as with GCU) to obtain
	// the offset and slope.
	//
	} else if (type.compare("SLF") == 0 && !isRecentGCUavailable) {

		if (resType.compare("L") == 0) {
			if (massType.compare("L") == 0) {

				// Row A
				linFitA = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
				if (numUmass > 3) {
					linFitA->startFit((long)slfLmLrA.size(), "SLF", "L", "L", 1);
					aA = linFitA->geta();
					bA = linFitA->getb();
					RcoeffA = linFitA->getCoefC();
					R2coeffA = linFitA->getCoefD();
					linFitA->errors("SLF", "L", "L", 1);
					sErrorA = linFitA->getSigEst();
					errOnaA = linFitA->getSa();
					errOnbA = linFitA->getSb();
					//printxFVector("m0",slfLmLrA);
				} else {
					cout << "Row A - LowRes/LowMass Only 1 mass seen" << singleM0 << endl;
				}

				// Row B
				linFitB = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
				if (numUmass > 3) {
					linFitB->startFit((long)slfLmLrB.size(), "SLF", "L", "L", 2);
					aB = linFitB->geta();
					bB = linFitB->getb();
					RcoeffB = linFitB->getCoefC();
					R2coeffB = linFitB->getCoefD();
					linFitB->errors("SLF", "L", "L", 2);
					sErrorB = linFitB->getSigEst();
					errOnaB = linFitB->getSa();
					errOnbB = linFitB->getSb();
					//printxFVector("m0",slfLmLrB);
				} else {
					cout << "Row B - LowRes/LowMass Only 1 mass seen" << singleM0 << endl;
				}

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;
			}

			// Write High Mass fit
			if (massType.compare("H") == 0) {

				// Row A
				linFitA = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
				if (numUmass > 2) {
					linFitA->startFit((long)slfHmLrA.size(), "SLF", "L", "H", 1);
					aA = linFitA->geta();
					bA = linFitA->getb();
					RcoeffA = linFitA->getCoefC();
					R2coeffA = linFitA->getCoefD();
					linFitA->errors("SLF", "L", "H", 1);
					sErrorA = linFitA->getSigEst();
					errOnaA = linFitA->getSa();
					errOnbA = linFitA->getSb();
					//printxFVector("m0",slfHmLrA);
				} else {
                    cout << "Row A - LowRes/HiMass Only 1 mass seen" << singleM0 << endl;
                }

				// Row B
				linFitB = new DFMS_LinFit_Class();
				numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
				if (numUmass > 2) {
					linFitB->startFit((long)slfHmLrB.size(), "SLF", "L", "H", 2);
					aB = linFitB->geta();
					bB = linFitB->getb();
					RcoeffB = linFitB->getCoefC();
					R2coeffB = linFitB->getCoefD();
					linFitB->errors("SLF", "L", "H", 2);
					sErrorB = linFitB->getSigEst();
					errOnaB = linFitB->getSa();
					errOnbB = linFitB->getSb();
					//printxFVector("m0",slfHmLrB);
				} else {
                    cout << "Row B - LowRes/HiMass Only 1 mass seen" << singleM0 << endl;
				}

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;

			}

			// If either of row A or B offsets are 0.0 then return failure
			if (aA == 0 || aB == 0) {
				return 0;
			}

			// If both row A or B offsets are NOT 0.0 then return success
			if (aA != 0 && aB != 0) {
				// Set Fit Quality ID flag
				if (fitRowsA > MIN4GCUFIT && R2coeffA < 0.90) {
					qualID = 2;
				} else if (fitRowsB > MIN4GCUFIT && R2coeffB < 0.90) {
					qualID = 2;
				} else {
					qualID = 0;
				}
				return 1;
			}

			// If either row A or B offsets are 0.0 then return success
			if (aA != 0 || aB != 0) {
				// Set Fit Quality ID flag
				if (fitRowsA > MIN4GCUFIT && R2coeffA < 0.90) {
					qualID = 2;
				} else if (fitRowsB > MIN4GCUFIT && R2coeffB < 0.90) {
					qualID = 2;
				} else {
					qualID = 0;
				}
				return 1;
			}

		} else if (resType.compare("H") == 0) {

			if (massType.compare("L") == 0) {

				// Row A
				linFitA = new DFMS_LinFit_Class();
                numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
                if (numUmass > 3) {
                    linFitA->startFit((long)slfLmHrA.size(), "SLF", "H", "L", 1);
                    aA = linFitA->geta();
                    bA = linFitA->getb();
                    RcoeffA = linFitA->getCoefC();
                    R2coeffA = linFitA->getCoefD();
                    linFitA->errors("SLF", "H", "L", 1);
                    sErrorA = linFitA->getSigEst();
                    errOnaA = linFitA->getSa();
                    errOnbA = linFitA->getSb();
                } else {
                    cout << "Row A - HiRes/LowMass Only 1 mass seen" << singleM0 << endl;
                }

				// Row B
				linFitB = new DFMS_LinFit_Class();
                numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
                if (numUmass > 3) {
                    linFitB->startFit((long)slfLmHrB.size(), "SLF", "H", "L", 2);
                    aB = linFitB->geta();
                    bB = linFitB->getb();
                    RcoeffB = linFitB->getCoefC();
                    R2coeffB = linFitB->getCoefD();
                    linFitB->errors("SLF", "H", "L", 2);
                    sErrorB = linFitB->getSigEst();
                    errOnaB = linFitB->getSa();
                    errOnbB = linFitB->getSb();
                } else {
                    cout << "Row B - HiRes/LowMass Only 1 mass seen" << singleM0 << endl;
                }

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;
			}

			// Write Medium Mass fit
			if (massType.compare("M") == 0) {

				// Row A
				linFitA = new DFMS_LinFit_Class();
                numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
                if (numUmass > 2) {
                    linFitA->startFit((long)slfMmHrA.size(), "SLF", "H", "M", 1);
                    aA = linFitA->geta();
                    bA = linFitA->getb();
                    RcoeffA = linFitA->getCoefC();
                    R2coeffA = linFitA->getCoefD();
                    linFitA->errors("SLF", "H", "M", 1);
                    sErrorA = linFitA->getSigEst();
                    errOnaA = linFitA->getSa();
                    errOnbA = linFitA->getSb();
                } else {
                    cout << "Row A - HiRes/MedMass Only 1 mass seen" << singleM0 << endl;
                }

				// Row B
				linFitB = new DFMS_LinFit_Class();
                numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
                if (numUmass > 2) {
                    linFitB->startFit((long)slfMmHrB.size(), "SLF", "H", "M", 2);
                    aB = linFitB->geta();
                    bB = linFitB->getb();
                    RcoeffB = linFitB->getCoefC();
                    R2coeffB = linFitB->getCoefD();
                    linFitB->errors("SLF", "H", "M", 2);
                    sErrorB = linFitB->getSigEst();
                    errOnaB = linFitB->getSa();
                    errOnbB = linFitB->getSb();
                } else {
                    cout << "Row B - HiRes/MedMass Only 1 mass seen" << singleM0 << endl;
                }

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;
			}

			// Write High Mass fit
			if (massType.compare("H") == 0) {

				// Row A
				linFitA = new DFMS_LinFit_Class();
                numUmass = uniqCommMasses(type, resType, massType, 1, &singleM0);
                if (numUmass > 2) {
                    linFitA->startFit((long)slfHmHrA.size(), "SLF", "H", "H", 1);
                    aA = linFitA->geta();
                    bA = linFitA->getb();
                    RcoeffA = linFitA->getCoefC();
                    R2coeffA = linFitA->getCoefD();
                    linFitA->errors("SLF", "H", "H", 1);
                    sErrorA = linFitA->getSigEst();
                    errOnaA = linFitA->getSa();
                    errOnbA = linFitA->getSb();
                } else {
                    cout << "Row A - HiRes/HiMass Only 1 mass seen" << singleM0 << endl;
                }

				// Row B
				linFitB = new DFMS_LinFit_Class();
                numUmass = uniqCommMasses(type, resType, massType, 2, &singleM0);
                if (numUmass > 2) {
                    linFitB->startFit((long)slfHmHrB.size(), "SLF", "H", "H", 2);
                    aB = linFitB->geta();
                    bB = linFitB->getb();
                    RcoeffB = linFitB->getCoefC();
                    R2coeffB = linFitB->getCoefD();
                    linFitB->errors("SLF", "H", "H", 2);
                    sErrorB = linFitB->getSigEst();
                    errOnaB = linFitB->getSa();
                    errOnbB = linFitB->getSb();
                } else {
                    cout << "Row B - HiRes/HiMass Only 1 mass seen" << singleM0 << endl;
                }

				delete linFitA; linFitA = 0;
				delete linFitB; linFitB = 0;

			}
		}

		// Set Fit Quality ID flag
		if (fitRowsA > MIN4GCUFIT && R2coeffA < 0.90) {
			qualID = 2;
		} else if (fitRowsB > MIN4GCUFIT && R2coeffB < 0.90) {
			qualID = 2;
		} else {
			qualID = 0;
		}

		// If either of row A or B offsets are 0.0 then return failure
		if (aA == 0 || aB == 0) {
			return 0;
		}

		return 1;
	}

	return 1;
}



//
// ---------------------- getFitFilesInDir -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method retrieves all the Fit Files in a directory
//
// inputs:

// returns:
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, July 2014
// =============================================================================
//
int DFMS_X0FitFile_Class::getFitFilesInDir(string sx0FitsDataPath, vector<fitFileInfo> &fitInfo) {

	string sFunctionName="DFMS_X0FitFile_Class::getFitFilesInDir";

  	DIR *dir;
  	struct dirent *entry;
	string fileName;
	fitFileInfo F;

	vector<string> fVec;


	if ((dir = opendir(sx0FitsDataPath.c_str())) == NULL) {
		sErrorMessage="Default Source directory - "+sx0FitsDataPath+" does not exist";
		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
  	} else {
  		while ((entry = readdir(dir)) != NULL) {
  			fileName = string(entry->d_name);
  			if (fileName.size() == 31) {
  				string prefix = fileName.substr(0,3);
  				string firstChar = fileName.substr(0,1);
  				// Skip all files shorter than 3 characters.. (like ./, ../)
  				if (firstChar.compare(".") != 0 && prefix.compare("X0_") == 0) {
  					fVec.push_back(fileName);
  				}
  			}
  		}
  		closedir(dir);

  		sort(fVec.begin(), fVec.end());

  		for(int i=0; i<fVec.size(); i++) {
  			string suffix = fVec[i].substr(fVec[i].find_last_of(".") + 1);
  			string thisType = fVec[i].substr(3,3);
  			string dateTime = fVec[i].substr(7,15);
  			string massType = fVec[i].substr(23,1);
  			string resType = fVec[i].substr(25,1);

  			// Only use non-empty .TAB files
  			if((suffix == "tab" || suffix == "TAB")) {
  				F.init();
  				F.fileName = fVec[i];
  				F.dateTime = dateTime;
  				F.type = thisType;
  				F.massType = massType;
  				F.resType = resType;
  				fitInfo.push_back(F);
  				//F.printData(thisType);
  			} else {
  				sErrorMessage="File: "+fVec[i]+" is not a candidate file";
  				writeToLog(sErrorMessage, sFunctionName, WARN);
  			}
  		}
  	}

	if (fitInfo.size() == 0) {
		sErrorMessage="Default source directory - " + sx0FitsDataPath+" does not contain expected files";
		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
	} else {
		sInfoMessage="Found "+util_intToString(fitInfo.size())+" X0 fit files in "+sx0FitsDataPath;
		writeToLog(sInfoMessage, sFunctionName, INFO);
		if(verbose >= 3) cout << sInfoMessage << endl;
	}

     return 1;
}

//
// ----------------------- findBestFitFile -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method finds the closest pix0 Fit File file to the current
// L2 file date and time
//
// inputs:
//   sx0FitsDataPath = Path to the x0 fit files
//   sRefDateAndTime - Input date and time
//   type - "GCU" or "SLF"
//   massType - "L" or "H"
//   fitInfo - A vector if all Fit files found
//
// outputs:
//   sBestFitFile - The best fit file matching the input criteria
//
// returns:
//   sBestMPST - best file name found
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, July 2014
// =============================================================================
//
int DFMS_X0FitFile_Class::findBestFitFile(string sx0FitsDataPath, string sRefDateAndTime,
								string type, string resType, string massType,
								vector<fitFileInfo> &fitInfo)
{

	string sFunctionName="findBestFitFile";

	string nBestDateAndTime = "0";
	string nDateAndTime;
	string nThisDateAndTime;
	string sDateAndTime;

	// Initially set to MPST
	nDateAndTime = sRefDateAndTime;
	sBestFitFile = "0";

	// Vector containing the specific type files needed
	vector<fitFileInfo> goodOnes;

	//cout << "type = " << type << " - resType = " << resType << " - massType = " << massType << endl;
	//cout << "fitInfo.size() = " << fitInfo.size() << endl;

	// Filter the fit files to keep only those that are of 'type' and 'massType'
	for (int i=0; i<fitInfo.size(); i=i+1) {
		//fitInfo[i].printData(type);
		//cout << "filename = " << fitInfo[i].fileName << endl;
		//cout << "type = " << fitInfo[i].type << " - resType = " << fitInfo[i].resType << " - massType = " << fitInfo[i].massType << endl;
		if ( (fitInfo[i].type).compare(type) == 0 &&
		     (fitInfo[i].massType).compare(massType) == 0 &&
		     (fitInfo[i].resType).compare(resType) == 0)   {
			goodOnes.push_back(fitInfo[i]);
		}
	}

	//cout << "goodOnes.size() = " << goodOnes.size() << endl;

	// If no type/resType/massType files available then use GCU
	if (goodOnes.size() == 0) {
		sInfoMessage="No X0 fit files for "+type+" - resType = "+resType+" - massType = "+massType+" ";
		sInfoMessage+=" use GCU";
		writeToLog(sInfoMessage, sFunctionName, INFO);
		cout << sInfoMessage << endl;
		return 0;
	}

	// Filter directory contents for those that match Mass Peak Search Table
    // file info
	if (goodOnes.size() == 1) {
		sBestFitFile = goodOnes[0].fileName;
		if (verbose >= 3) {
			cout << "sBestFitFile = " << sBestFitFile << endl;
        }
	} else {
		sBestFitFile = goodOnes[0].fileName;
		for (int i=0; i<goodOnes.size(); i++) {
			sDateAndTime = goodOnes[i].dateTime;

			nThisDateAndTime = sDateAndTime;

			//cout << "sRefDateAndTime = " << sRefDateAndTime << endl;
			//cout << "nThisDateAndTime = " << nThisDateAndTime << endl;
			//cout << "sDateAndTime = " << sDateAndTime << endl;

			int comp1 = nThisDateAndTime.compare(nDateAndTime);
			int comp2 = nThisDateAndTime.compare(nBestDateAndTime);
			if (comp1 < 0 && comp2 > 0) {
				nBestDateAndTime = nThisDateAndTime;
				sBestFitFile = goodOnes[i].fileName;
				//cout << "sBestFitFile = " << sBestFitFile << endl;
				if (verbose >= 3) {
					cout << "sBestFitFile = " << sBestFitFile << endl;
				}
			}
		}
	}

	// found no matching files
	if (sBestFitFile == "0") {
		sErrorMessage = "No suitable X0 fit file ";
		sErrorMessage += "was found in supplied path: " + sx0FitsDataPath;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	} else {
		//cout << "type = " << type << " - sRefDateAndTime = " << sRefDateAndTime << " -  sBestFitFile = " << sBestFitFile << endl;
	}

	return 1;

}

//
// ------------------------------ printPix0File ------------------------------------
//
void DFMS_X0FitFile_Class::printPix0File(string type) {

	// First print the PDS Header

	map<string,string>::const_iterator it;

	if (type == "PDSHeader" || type == "All") {
		for (it = PDSheader.begin(); it != PDSheader.end(); ++it) {
			printf("%-32s =     %-s\n",(it->first).c_str(),(it->second).c_str());
		}
		cout << endl;
	}

	// Next print out Cal Object description
	if (type == "fitObjA" || type == "All") {
		for (it = fitObjA.begin(); it != fitObjA.end(); ++it) {
			printf("%32s =     %-s\n",(it->first).c_str(),(it->second).c_str());
		}
		cout << endl;
	}

	// Next print out Cal Object description
	if (type == "fitObjB" || type == "All") {
		for (it = fitObjB.begin(); it != fitObjB.end(); ++it) {
			printf("%32s =     %-s\n",(it->first).c_str(),(it->second).c_str());
		}
		cout << endl;
	}

	// Next print out Data Object description
	if (type == "fitResDataObj" || type == "All") {
		for (it = fitResDataObj.begin(); it != fitResDataObj.end(); ++it) {
			printf("%32s =     %-s\n",(it->first).c_str(),(it->second).c_str());
		}
	     cout << "END" << endl;
	}

	if (type == "fitDataA" || type == "All") {
		for (int i=0; i<fitRowsA; i++) {
			printf("%6s, %6s, %6s, %6s, %29s,\n",
				(fitDataA[i][0]).c_str(), (fitDataA[i][1]).c_str(), (fitDataA[i][2]).c_str(),
				(fitDataA[i][3]).c_str(), (fitDataA[i][4]).c_str(), (fitDataA[i][5]).c_str());
		}
	}

	if (type == "fitDataB" || type == "All") {
		for (int i=0; i<fitRowsB; i++) {
			printf("%6s, %6s, %6s, %6s, %29s,\n",
				(fitDataB[i][0]).c_str(), (fitDataB[i][1]).c_str(), (fitDataB[i][2]).c_str(),
				(fitDataB[i][3]).c_str(), (fitDataB[i][4]).c_str(), (fitDataB[i][5]).c_str());
		}
	}

	// Finally print out the data
	if (type == "pix0Data" || type == "All") {
			printf("%10.4e, %10.4e, %10.4e, %10.4e, %10.4e, %10.4e, %10.4e, %10.4e,\n",
					fitResData[0], fitResData[1], fitResData[2], fitResData[3],
					fitResData[4], fitResData[5], fitResData[6], fitResData[7]);
	}

}

//
// --------------------- determinePix0 ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Find pix0 for non-GCU files using a polynomial fit
//
// inputs:
// 		a      - known value of offset (for Row A or B)
//		b      - known slope of earlier fit (for Row A or B)
//      tMass  - independent variable m (mass)
//
// returns:
//   pix0
// =============================================================================
// History: Written by Mike Rinaldi, July 2104
// =============================================================================
//
double DFMS_X0FitFile_Class::determinePix0(double a, double b, double tMass) {

    double pix0;
    pix0 = a + b*tMass;

    return pix0;
}

//
// --------------------- determineA ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Find a from pix0,b and m  (for Row A or B)
//
// inputs:
// 		pix0 - known value of pix0
//		b    - known slope of earlier fit
//      m    - independent variable m (mass)
// returns:
//   a
// =============================================================================
// History: Written by Mike Rinaldi, July 2014
// =============================================================================
//
double DFMS_X0FitFile_Class::determineA(double pix0, double b, double m) {

    double a;
    a = pix0 - b*m;

    return a;
}

//
// ---------------------- uniqCommMasses -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method returns the number of unique commanded masses in a vector
//
// inputs:
//    type -  whether GCU or SLF
//    resType - resolution whether "L" or "H"
//    massType - mass whether "L" or "H"
//    iRow - Rows A/B
//
// output:
//    muse - m0 that is
//
// returns:
//   returns number unique values
// =============================================================================
// History: Written by Mike Rinaldi, July 2014
// =============================================================================
//
int DFMS_X0FitFile_Class::uniqCommMasses(string type, string resType, string massType,
											int iRow, double *muse) {

	vector<double> m0;

	if (type.compare("GCU") == 0) {
		if (resType.compare("L") == 0) {
			if (iRow ==1) {
				if (massType.compare("L") == 0) {
					for (int i=0; i<gcuLmLrA.size(); i++) m0.push_back(gcuLmLrA[i].m0);
					if (numUnique(m0) == 1) *muse = gcuLmLrA[0].m0;
				} else if (massType.compare("H") == 0){
					for (int i=0; i<gcuHmLrA.size(); i++) m0.push_back(gcuHmLrA[i].m0);
					if (numUnique(m0) == 1) *muse = gcuHmLrA[0].m0;
				}
			} else {
				if (massType.compare("L") == 0) {
					for (int i=0; i<gcuLmLrB.size(); i++) m0.push_back(gcuLmLrB[i].m0);
					if (numUnique(m0) == 1) *muse = gcuLmLrB[0].m0;
				} else if (massType.compare("H") == 0){
					for (int i=0; i<gcuHmLrB.size(); i++) m0.push_back(gcuHmLrB[i].m0);
					if (numUnique(m0) == 1) *muse = gcuHmLrB[0].m0;
				}
			}
		} else if (resType.compare("H") == 0) {
			if (iRow == 1) {
				if (massType.compare("L") == 0) {
					for (int i=0; i<gcuLmHrA.size(); i++) m0.push_back(gcuLmHrA[i].m0);
					if (numUnique(m0) == 1) *muse = gcuLmHrA[0].m0;
				} else if (massType.compare("M") == 0){
					for (int i=0; i<gcuMmHrA.size(); i++) m0.push_back(gcuMmHrA[i].m0);
					if (numUnique(m0) == 1) *muse = gcuMmHrA[0].m0;
				} else if (massType.compare("H") == 0){
					for (int i=0; i<gcuHmHrA.size(); i++) m0.push_back(gcuHmHrA[i].m0);
					if (numUnique(m0) == 1) *muse = gcuHmHrA[0].m0;
				}
			} else {
				if (massType.compare("L") == 0) {
					for (int i=0; i<gcuLmHrB.size(); i++) m0.push_back(gcuLmHrB[i].m0);
					if (numUnique(m0) == 1) *muse = gcuLmHrB[0].m0;
				} else if (massType.compare("M") == 0){
					for (int i=0; i<gcuMmHrB.size(); i++) m0.push_back(gcuMmHrB[i].m0);
					if (numUnique(m0) == 1) *muse = gcuMmHrB[0].m0;
				} else if (massType.compare("H") == 0){
					for (int i=0; i<gcuHmHrB.size(); i++) m0.push_back(gcuHmHrB[i].m0);
					if (numUnique(m0) == 1) *muse = gcuHmHrB[0].m0;
				}
			}
		}
	} else if (type.compare("SLF") == 0) {
		if (resType.compare("L") == 0) {
			if (iRow ==1) {
				if (massType.compare("L") == 0) {
					for (int i=0; i<slfLmLrA.size(); i++) m0.push_back(slfLmLrA[i].m0);
					if (numUnique(m0) == 1) *muse = slfLmLrA[0].m0;
				} else if (massType.compare("H") == 0){
					for (int i=0; i<slfHmLrA.size(); i++) m0.push_back(slfHmLrA[i].m0);
					if (numUnique(m0) == 1) *muse = slfHmLrA[0].m0;
				}
			} else {
				if (massType.compare("L") == 0) {
					for (int i=0; i<slfLmLrB.size(); i++) m0.push_back(slfLmLrB[i].m0);
					if (numUnique(m0) == 1) *muse = slfLmLrB[0].m0;
				} else if (massType.compare("H") == 0){
					for (int i=0; i<slfHmLrB.size(); i++) m0.push_back(slfHmLrB[i].m0);
					if (numUnique(m0) == 1) *muse = slfHmLrB[0].m0;
				}
			}
		} else if (resType.compare("H") == 0) {
			if (iRow == 1) {
				if (massType.compare("L") == 0) {
					for (int i=0; i<slfLmHrA.size(); i++) m0.push_back(slfLmHrA[i].m0);
					if (numUnique(m0) == 1) *muse = slfLmHrA[0].m0;
				} else if (massType.compare("M") == 0){
					for (int i=0; i<slfMmHrA.size(); i++) m0.push_back(slfMmHrA[i].m0);
					if (numUnique(m0) == 1) *muse = slfMmHrA[0].m0;
				} else if (massType.compare("H") == 0){
					for (int i=0; i<slfHmHrA.size(); i++) m0.push_back(slfHmHrA[i].m0);
					if (numUnique(m0) == 1) *muse = slfHmHrA[0].m0;
				}
			} else {
				if (massType.compare("L") == 0) {
					for (int i=0; i<slfLmHrB.size(); i++) m0.push_back(slfLmHrB[i].m0);
					if (numUnique(m0) == 1) *muse = slfLmHrB[0].m0;
				} else if (massType.compare("M") == 0){
					for (int i=0; i<slfMmHrB.size(); i++) m0.push_back(slfMmHrB[i].m0);
					if (numUnique(m0) == 1) *muse = slfMmHrB[0].m0;
				} else if (massType.compare("H") == 0){
					for (int i=0; i<slfHmHrB.size(); i++) m0.push_back(slfHmHrB[i].m0);
					if (numUnique(m0) == 1) *muse = slfHmHrB[0].m0;
				}
			}
		}
	}
	int numU = numUnique(m0);
	m0.clear();
	return numU;

}


//
// ---------------------- getPix0Unc -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method calculates the pix0's for the mass m0 for all combinations of
// a/b error. Then it returns the maximum difference of all the resultant pix0's
//
// inputs:
//
// returns:
//   max Diff of Pix0
// =============================================================================
// History: Written by Mike Rinaldi, August 2014
// =============================================================================
//
double DFMS_X0FitFile_Class::getPix0Unc(int iRow, double m0) {

	vector<double> pix0s;

	if (iRow == 1) {
		pix0s.push_back((aA-errOnaA) + (bA-errOnbA)*m0);
		pix0s.push_back((aA-errOnaA) + (bA+errOnbA)*m0);
		pix0s.push_back((aA+errOnaA) + (bA-errOnbA)*m0);
		pix0s.push_back((aA+errOnaA) + (bA+errOnbA)*m0);
	} else {
		pix0s.push_back((aB-errOnaB) + (bB-errOnbB)*m0);
		pix0s.push_back((aB-errOnaB) + (bB+errOnbB)*m0);
		pix0s.push_back((aB+errOnaB) + (bB-errOnbB)*m0);
		pix0s.push_back((aB+errOnaB) + (bB+errOnbB)*m0);
	}
	double min_pix0 = *min_element(pix0s.begin(), pix0s.end());
	double max_pix0 = *max_element(pix0s.begin(), pix0s.end());

	double max_diff = max_pix0 - min_pix0;

	return max_diff;
}

//
// ---------------------- getL2header -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Get the L2 Header
//
// inputs:
//   path  -  path to L2
//   file  -  L2 file name
// returns:
//   1 for success, 0 for failure
// =============================================================================
// History: Written by Mike Rinaldi, April 2015
// =============================================================================
//
int DFMS_X0FitFile_Class::getL2header(string path, string L2File) {

	string sFunctionName = "DFMS_X0FitFile_Class::getL2header";

	fstream is;
    int nSuccess;

    DFMS_PDS_file_Class *PDSObj = new DFMS_PDS_file_Class();

	nSuccess = PDSObj->openFile(path+L2File,is,1);
	if (!nSuccess) {
		sErrorMessage = "Unable to open L2 PDS file: ";
		sErrorMessage += L2File+" - Skipping file processing";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		if (is.is_open()) is.close();
		delete PDSObj;
		return 0;
	}

    // Read the L2 file PDS header
	nSuccess = PDSObj->readPDSheader(is);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the PDS header for file: ";
		sErrorMessage += L2File+" - Skipping file processing";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	// Save the L2header
	L2header.insert(PDSObj->PDSheader.begin(), PDSObj->PDSheader.end());

    // Close the L2 File stream
	PDSObj->closeFile(is);

    // delete PDS object
	delete PDSObj;

	return 1;

}
