/* 
 * File:   main.c
 * Author: marinaldi
 *
 * Created on January 19, 2013, 11:08 AM
 */

#include "DFMS_process_L2_to_L3.hh"

// ------------------------ Global Variables -----------------------------------

// Step 00 - Set up global and then local (after main() ) variables

int verbose=0;
int numL2Files=1;

// Software Info
string sErrorMessage;
string sInfoMessage;
string sCurrentSoftwareName = "DFMS_PDS_L2_TO_L3";
string sCurrentSoftwareVersionID = "2018-7-22,v1.90";

// Default names of the configuration files
string cfgFile = "ConfigFile.dat";   // DFMS_PDS_L2_to_L3 configuration file
string constantsFile = "DFMS_Constants.dat";   // DFMS related constants
string exclusionFile = "DFMS_Exclusion_Times.dat";   // Data exclusion times

// Global Log file streams
fstream toDebug;
fstream toLog;
fstream l3Log;
fstream QuickLookLog;
fstream pix0D;

// Use -p CLA when running to set this to 1. Default is 0 or no plots
int iPltFile=0;

// True if processing single Mode files
int intMode = 0;

// temporary globals
int gcuLmLrGoodFit=0;
int gcuHmLrGoodFit=0;
int gcuLmHrGoodFit=0;
int gcuMmHrGoodFit=0;
int gcuHmHrGoodFit=0;
int slfLmLrGoodFit=0;
int slfHmLrGoodFit=0;
int slfLmHrGoodFit=0;
int slfMmHrGoodFit=0;
int slfHmHrGoodFit=0;

// Special masses to use for keeping Peak Info Time Series results
int spMasses[NUMSPECIALMASSES] = {13, 14, 15, 16, 17, 18, 19, 20,
		                          21, 22, 23, 24, 25, 26, 27, 28, 29, 30,
		                          31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
		                          41, 42, 43, 44, 45, 46, 47, 48, 49, 50,
		                          51, 52, 53, 54, 55, 56, 57, 58, 59, 60,
		                          61, 62, 63, 64, 65, 66, 67, 68, 69, 70,
  	  	  	  	  	  	  	  	  71, 72, 73, 74, 75, 76, 77, 78, 79, 80,
  	  	  	  	  	  	  	  	  81, 82, 83, 84, 85, 86, 87, 88, 89, 90,
  	  	  	  	  	  	  	  	  91, 92, 93, 94, 95, 96, 97, 98, 99,100,
  	  	  	  	  	  	  	  	 101,102,103,104,105,106,107,108,109,110,
  	  	  	  	  	  	  	  	 111,112,113,114,115,116,117,118,119,120,
  	  	  	  	  	  	  	  	 121,122,123,124,125,126,127,128,129,130,
  	  	  	  	  	  	  	  	 131,132,133,134,135,136,137,138,139,140};

// Instrument Mode emissivity
double emission = 0.0;

// STL container to hold all SLF masses in a unique ordered set
set<double> slfSet;

// STL container to hold number commanded masses fit/failed
map<double,int> thisRunCommMassFitSuccess;
map<double,int> thisRunCommMassFitFailed;

// Note that the assigned values below are initial values and will
// probably be modified by values in the ConfigFile.dat file.
// The general rule is to set these values in the ConfigFile.dat
string sInstallDir="/Users/marinaldi/ROSINA/workspace/DFMS/src/";
string sLogPath="logs/";
string sDiagPath="data/plotFiles/";
string sTimeSeriesPath="timeSeries/";
string sL3InfoLogPath="L3InfoLogs/";
string sQLLogPath="QuickLook/";
string sL2sourcePath="data/sL2sourcePath/";
string sL3outputPath="data/sL3outputPath/";
string sPlotFilePath="data/sPlotFilePath/";
string sModeIDTablePath="tables/ModeID/";
string sModeIDTableFilename= "MODE_ID_TABLE_20100102.TAB";
string sGainTablePath="tables/Gain/";
string sMassPeakTablePath= "tables/MassPeak/";
string sCOPSdataPath= "data/sCOPSdataPath/";
string sCOPSdataFilename= "ng_20090218_150425579_m0322.tab";
string sInstrumentModel = "FM";
string sGainTableBaseName="DFMS_GainTable_";
string sPixelGainTableBaseName="DFMS_PixelGainTable_";
string sx0FitsDataPath="data/sx0FitsDataPath/";
string sGNUplotPath="/opt/local/bin/";
string sPix0DFilePath="pix0D/";
string sPeakExclusionFile1="DFMS_20140401-20160127_Peak_Exclusion.dat";
string sPeakExclusionFile2="DFMS_20160127-20161001_Peak_Exclusion.dat";
string pgtPath="";
string gtPath="";
// The name of the best MPST/COPS files associated with a specific L2 file
string sBestMPST="";
string sBestNGCOPS="";
string sBestRGCOPS="";
string sBestBGCOPS="";
int DIAGTOFILE = 1;                // Default is to write L2/L3 data/fit to file
int L3INFOTOFILE = 0;              // Default is to not write L# Info files
int QLINFOFILE = 1;                // Default is to open quicklook file
int PIX0DELTOFILE = 1;             // Will create and/or append to the file DFMS_pix0DeltaFile.log

string sBestFitFile;

// Best Pixel Gain Table time
string sBestPGTtime;
int NUMPG=16;
int debug = 0;

// Various constants.  some of these are set/modified in the
// file DFMS_FM_Constants.dat therefore these are just default values
double GCUMASSDEL = 1.0;
double NONGCUMASSDEL = 1.0;
double GCULOWCUTOFF = 30.0;		// Low mass cutoff for GCU files
double GCUHICUTOFF = 70.0;		// High mass cutoff for GCU files
double SLFLOWCUTOFF = 30.0;		// Low mass cutoff for SLF files
double SLFHICUTOFF = 70.0;		// High mass cutoff for SLF files
double PPMDIFFMIN = 500.0;  	// Variance from known mass
double ENOISET = 0.1;
double STDCALDEV = 0.01;		// Fixed value for now
double MODAMP = 45;				// Row B Offset modulation Amplitude
double PKHEIGHTLIMIT = 100;		// Base peak high limit
double HRMASSDELTA = 0.1;  		// Force a +/- 0.1 mass error on high res modes
double HRPIX0UNC = 20.0;  		// Base pix0 uncertainty for high res data
double LRPIX0UNC = 5.0;   		// Base pix0 uncertainty for low res data
int NUMSDEV = 2;				// Number of std Devs above meanOffset over which to look for peaks
int TIMESERIESOUT = 0;          // Default is to not write Time history variables
double BADDATA = -999.0;  		// Bad Data Flag
bool CONVERTOUTBPEAK = true;  	// Do the L2->L3 Conversion for out of bounds peaks
bool CONVERTLOWAMP = true;    	// Do the L2->L3 Conversion for peaks whose height is < 1.1*offset
bool CONVERTHIGHRES = true;    	// Convert high res data using low res technique
int PKLOWPT = 100;     			// Default smallest pixel position to consider for mass peak discovery
int PKHIGHPT = 400;    			// Default largest pixel position to consider for mass peak discovery
int MIN4GCUFIT = 4;				// Default minimum number of GCU files to necessary perfrom linear fit
int MIN4SLFFIT = 3;	    		// Default minimum number of SLF files to necessary perfrom linear fit
int NUMDAYSTOFITGCU = 3;		// Number of Days to include into one pix0 Fit file for GCU
int NUMDAYSTOFITSLF = 1;		// Number of Days to include into one pix0 Fit file for SLF
double Q = 1.602176565E-19; 	// Electron charge in Coulombs
double CLEDA = 4.22E-12;  		// LEDA capacitance
double CONVADC = 6.105E-4; 		// ADC conversion constant
double DISP = 127000.0;			// Dispersion constant
double CONV = 25.0;     		// conversion const pix to um
int COPSDELTA = 10;     		// COPS time difference between L2 and COPS files (in minutes)
double PIX0ROWDIFF = 6.0;       // Sets a Row A/B pix0 delta. The value of this delta printed
                                // to file pix0D/DFMS_pix0DeltaFile.log contains an "*" is the
                                // delta is > PIX0ROWDIFF
string sRow[2] = {"A","B"};

string bestFitFile="";

// C++ STL Map variables to track reasons an L2 file was exceptetional or skipped
map<string,int> sideAexceptionType;
map<string,int> sideBexceptionType;
map<string,int> exceptionType;
map<string,int> skipReas;
map<string,int> pix0Spread;
// C++ STL Map holding the modes processed during this run
map<int,int> modesPresent;

// C++ STL Map variable holding Mode information
map<int,DFMSModeInfo> modeData;

// Logicals that are set if Error conditions exist in peak fitting Class
bool infNanErr = false;
bool stepFuncA = false;
bool stepFuncB = false;

// File processing stats counters
int totFilesSkipped=0;
int numVerificPeaksFound=0;
int numVerificPeaksUsed=0;

// C++ STL Vectors holding GCU,Pix0 mass tables for fitting purposes
vector<x0FitInfo> gcuLmLrA, gcuHmLrA, gcuLmLrB, gcuHmLrB;
vector<x0FitInfo> gcuLmHrA, gcuMmHrA, gcuHmHrA, gcuLmHrB, gcuMmHrB, gcuHmHrB;
// C++ STL Vectors holding SLF,Pix0 mass tables for fitting purposes
vector<x0FitInfo> slfLmLrA, slfHmLrA, slfLmLrB, slfHmLrB;
vector<x0FitInfo> slfLmHrA, slfMmHrA, slfHmHrA, slfLmHrB, slfMmHrB, slfHmHrB;

// C++ STL PPMdiff vectors to be used for nonGCU/nonSLF ppmdiff estimation
vector<ppmInfo> ppmSLFLmLrA, ppmSLFHmLrA, ppmSLFLmLrB, ppmSLFHmLrB;
vector<ppmInfo> ppmSLFLmHrA, ppmSLFMmHrA, ppmSLFHmHrA, ppmSLFLmHrB, ppmSLFMmHrB, ppmSLFHmHrB;

// C++ STL QuickLook Stats variables
pair<double,double> pkPPMpair;
vector<pair<double,double> > pkPPMvec;

// C++ STL Define a map of species mass in amu versus species ID
map<double,spcInfo> speciesDef;

// The name of a specific L2 file to process (from -f command line arg)
string dfmsFileName;

// this flag is set by CLA if processing one file
int doOneFile=0;

// Flag used to override the configuration value of
//sL2sourcePath and sL3outputPath
int overRideInput=0;
int overRideOutput=0;

// CLA flag set to Dump L2 order to file
int dumpL2Order = 0;

// If this flag is set it will process new GCU/SLF files so
// that new X0Fit files will be created. Default is 1
int processNewGCUSLFfiles = 1;

// C++ STL Define the number of distinct modes per run.
// Also map to set these to true
map<int,bool> distinctModes;
map<int,polyInfo> pC;

// Initial L2 Julian times
double L2TimeMax = -1.0e99;
double L2TimeMin = 1.0e99;

// User supplied pix0 for High res processing (not implemented)
int useExternalPix0 = 0;

// The user supplied external pix0 values.
// These are set when running the code with -x argument
// (not implemented)
double ePix0A = 0;
double ePix0B = 0;

// QuickLook print counter
int qlCount=0;

// Flag to set pre/post GCU availability.  This is needed to change the code
// behavior since the GCU mechanism failed as of 12/27/2014.
bool isRecentGCUavailable=true;

// Flag set to true for all dates after 20140101T00:00:00
bool isNewPDS=false;

// These are set to 1 if their are 2 or more GS PG Tables
// 0 otherwise
int multiGS[16] = {0};

// The Level 3 File information object.  This is the main object holding
// much of the calculated info for each L2 file.  A new object is created
// each time an L2 file is processed.  The previous is deleted.
DFMS_L3Info_Class *L3Info;

// Define a vector to hold exclusion time/comment info
vector<excludeInfo> dateExclusion;

// Peaks to exclude
map<int,vector<pair<int,int> > > peaksToExclude1;
map<int,vector<pair<int,int> > > peaksToExclude2;
string sPeakExclFile1Date;
string sPeakExclFile2Date;


// End of Line variable
char EOL[] = {static_cast<char>('\r\n')};
string dfmsEOL;

// CEM Variables
double CEMHR = 10000.0;   // delM/M0 constant
double CEMLR = 1000.0;    // delM/M0 constant

// Do CEM files only
bool DOCEM = false;

// Set L3 Mass precision (1 = 4 digits after comma, 0 = 2 digits after comma
int L3SET_MASS_PRECISION = 1;

int iicnt = 0;

bool debugLog=false;

//
//   --------------------- Begin main DFMS_process_L2_to_L3 driver code -------------------
//
int main(int argc, char** argv) {

	//raise(SIGTRAP);
    
	// set up local variables
    string sFunctionName="DFMS_process_L2_to_L3";

	int nSuccess = 0;
	string file;
	string L2File, L3File;
    int totFilesAttempted=0;
    int totFilesSuccessfullyProcessed=0;
    int begId=0, endId=0;

    // Various reasons to skip L2 file processing
    skipReas["notMCPorCEM"] = 0;
    skipReas["peakHeightLimit"] = 0;
    skipReas["highRes"] = 0;
    skipReas["rosMassZero"] = 0;
    skipReas["gainNotDef"] = 0;
    skipReas["noMPSTfile"] = 0;
    skipReas["gcuMassNotInMPST"] = 0;
    skipReas["curvFitBad"] = 0;
    skipReas["pix0Bounds"] = 0;
    skipReas["peakOutOfBounds"] = 0;
    skipReas["L2HKLineOverun"] = 0;
    skipReas["fileExcluded"] = 0;
    skipReas["illegalCEMmass"] = 0;

    // Counters for various exceptions
    sideAexceptionType["peakHeightLimit"] = 0;
    sideBexceptionType["peakHeightLimit"] = 0;
    sideAexceptionType["peakOutOfBounds"] = 0;
    sideBexceptionType["peakOutOfBounds"] = 0;
    sideAexceptionType["peakFinderError"] = 0;
    sideBexceptionType["peakFinderError"] = 0;
    exceptionType["highRes"] = 0;

    // Define an array to hold the SLF (SeLF calibration peak masses).
    // Note:  To add or remove members from this array you will need to adjust the
    //        value of NUMSLFPKMASSES accordingly.
    //        NUMSLFPKMASSES is found in DFMS_definedConstants.hh
    //
    double slfMasses[NUMSLFPKMASSES] =
    {
    13,         // CH
    15,         // CH3
    15.01,      // CH3
    16,         // O16
    16.51,      // O16
    18,         // H2O
    18.16,      // H2O
    21.98,      // CO^2++
    22,         // CO^2++
    24.0,       // C2
    24.18,      // C2
    25,         // C2H
    35.4,       // C3H
    37,         // C3H
    38.94,      // C3H3
    39,         // C3H3
    42.83,      // CO2
    44,         // CO2
    47.12,      // SO
    48,         // SO
    50,         // C4H2
    51,         // C4H3
    51.83,      // C4H5
    53,         // C4H5
    60,         // OCS
    62.72,      // OCS
    75.89,      // C6H6
    76,         // CS2
    78.0,       // C6H6
    91,         // C7H7
    91.83       // C7H7
    };

    // Put self calibration masses into a self-sorted C++ STL set container
    for (int i=0; i<NUMSLFPKMASSES; i++) slfSet.insert(slfMasses[i]);

    // Pixel Gain Table File time info
    vector<DirInfo> gtTimeInfo,PGTtimeInfo;
    vector<pgFileInfo> PGTFileInfo;
    vector<gtFileInfo> GTFileInfo;

    // MPST and COPS files Info
    vector<L2FileInfo> gcuMPSTinfo;
    vector<L2FileInfo> nonGCUMPSTinfo;
    vector<L2FileInfo> COPSNGinfo;
    vector<L2FileInfo> COPSRGinfo;
    vector<L2FileInfo> COPSBGinfo;
     
    // Various objects needed for processing
    DFMS_ModeID_Table_Class *modeIDobject;      // Object with Mode Table methods/data
	DFMS_PDS_L2_dir_Class *L2DirObj;            // Object with L2 directory methods/data
	DFMS_PDS_L2_file_Class *L2Obj;              // Object with specific L2 file methods/data
	DFMS_PDS_L3_file_Class *L3Obj;              // Object with specific L3 file methods/data
	DFMS_ProcessMCP_Class *mcpL2toL3;           // Object that performs MC file L2->L3 conv. methods/data
	DFMS_ProcessCEM_Class *cemL2toL3;           // Object that performs CEM file L2->L3 conv. methods/data
	DFMS_PixelGain_Interp_Class *pgObj;         // Object with Pixel Gain interpolated methods/data
    DFMS_GainTable_Interp_Class *gtObject;      // Object with Gain Table interpolated methods/data
     
    // Create the Mode ID structure
    DFMSModeInfo modeInfo;

	//  ------------------------------- Begin Run Setup ---------------------------------

    // Get seconds of CPU time at start of processing
    string timeAtStart = getProcessTime(4);

    // Get date/time at start
    string sTstart = getProcessTime(3);

    // Get actual time in milli-seconds at start of processing
    const long double sysTimeMS1 = time(0)*1000;

    // Step 1 - Process any CLA's
  	// Get CLA's (Command Line Arguments - if any) info
	DFMS_process_CLA(argc, argv);

    // Step 2 - Read Configuration File with Default run values
	nSuccess = DFMS_readCfgFile(cfgFile);
    if (!nSuccess) {
		sErrorMessage = "Unable to read required Configuration file: "+cfgFile;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		cout << "Unable to read required Configuration file: " << cfgFile << endl;
		exit(EXIT_FAILURE);
	}

    // Step 3 - Read DFMS Constants File with Default run values
	nSuccess = DFMS_readConstFile(constantsFile);
    if (!nSuccess) {
		sErrorMessage = "Unable to read required DFMS Constants file: "+constantsFile;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		cout << "Unable to read required DFMS Constants file: " << constantsFile << endl;
		exit(EXIT_FAILURE);
	}

    // Read Excluded region file.  This file contains date/time info pertaining to
    // periods when for various reasons L2 files should not be processed
    nSuccess = DFMS_readExclusionFile();
    if (!nSuccess) {
		sInfoMessage = "Unable to read exclusion times file: "+exclusionFile;
		writeToLog(sInfoMessage, sFunctionName, INFO);
		cout << "Unable to read exclusion times file: " << exclusionFile << endl;
	}
    
    // Read both peak Exclusion files.  These files contain the peak boundaries
    // required by the Offset fitting function.
    nSuccess = DFMS_readPeakExclusionFiles();
    if (!nSuccess) {
        sErrorMessage = "Unable to read either peak exclusion info file: "+sPeakExclusionFile1;
        sErrorMessage += " or "+sPeakExclusionFile2;
        writeToLog(sErrorMessage, sFunctionName, INFO);
        cout << sErrorMessage << endl;
        exit(EXIT_FAILURE);
    }
    
    // Open the Debug Logger stream for writing
    if (debugLog) {
        string DebugLogFile = "/Users/marinaldi/xcode/debug/DFMS_DebugLogFile-"+getProcessTime(2)+".log";
        toLog.open(DebugLogFile.c_str(), fstream::out | fstream::binary);
    }
    
    // Open pix0Deltas file
    if (PIX0DELTOFILE) {
        string pix0DFile = sPix0DFilePath+"DFMS_pix0DeltaFile.log";
        pix0D.open(pix0DFile.c_str(), fstream::out | fstream::binary | fstream::app);
    }
    
    // Open the Run Logger stream for writing
    string LogFile = sLogPath+"DFMS_processLogFile-"+getProcessTime(2)+".log";
    toLog.open(LogFile.c_str(), fstream::out | fstream::binary);

    // Open <optional> L3InfoLog stream for writing
	if (L3INFOTOFILE) {
		string L3LogFile = sL3InfoLogPath+"DFMS_L3InfoLogFile-"+getProcessTime(2)+".log";
		l3Log.open(L3LogFile.c_str(), fstream::out | fstream::binary);
	}

    // Open <optional> Quick Look log stream for writing
	if (QLINFOFILE) {
		string QLFile = sQLLogPath+"DFMS_QuickLookLogFile-"+getProcessTime(2)+".log";
		QuickLookLog.open(QLFile.c_str(), fstream::out | fstream::binary);
	}

    // Check OS type and set the End-Of-Line character(s) accordingly.
    // The variable used, checkOStype,by checkOStype()is defined in DFMS_definedConstants.hh
    checkOStype();

    // Define some known mass/species pairs.  This is used mainly for naming
    // specific known peaks. This file contains MPST named peaks and others not in the MPST.
    DFMS_speciesTable();

    // Optionally (user defined) open a Quick Look file to store m0,pix0,ppmDiff vs mode/filename
	if (QLINFOFILE) {
		QuickLookLog << dfmsEOL;
		QuickLookLog << "------------------------------------------------------------------------------------------------------------------------------------" << dfmsEOL;
		QuickLookLog << "   --------------------------------------- Quick Look Log at: "<< getProcessTime(1) << " --------------------------------------" << dfmsEOL;
		QuickLookLog << "--------------------------------------------------------------------------------------------------------------------------------------" << dfmsEOL << dfmsEOL;
		qlHeader();
	}

	// Optionally  (user defined) open L3Info Log file
	if (L3INFOTOFILE) {
		l3Log << dfmsEOL;
	    l3Log << "----------------------------------------------------------------------------" << dfmsEOL;
	    l3Log << "   ----- Begin Logging current L2 to L3 process at: "<< getProcessTime(1) << "------" << dfmsEOL;
	    l3Log << "----------------------------------------------------------------------------" << dfmsEOL;
	    l3Log << "                      Inputs from Configuration file: " << cfgFile << dfmsEOL;
	    l3Log << "                     Inputs DFMS Constants from file: " << constantsFile << dfmsEOL;
	    l3Log << "                     DFMS L2 to L3 install directory: " << sInstallDir << dfmsEOL;
	    l3Log << "                               L3 Info Log File Path: " << sL3InfoLogPath << dfmsEOL;
	    l3Log << "                                       Log File Path: " << sLogPath << dfmsEOL;
	    l3Log << "                                      L2 Source Path: " << sL2sourcePath << dfmsEOL;
	    l3Log << "                                      L3 Output Path: " << sL3outputPath << dfmsEOL;
	    l3Log << "                                      MPS Table Path: " << sMassPeakTablePath << dfmsEOL;
	    l3Log << "                                     Gain Table Path: " << sGainTablePath << dfmsEOL;
	    l3Log << "                                      COPS Data Path: " << sCOPSdataPath << dfmsEOL;
	    l3Log << "                                      Plot File Path: " << sPlotFilePath << dfmsEOL;
	    l3Log << "                           Diagnostic Plot Data Path: " << sDiagPath << dfmsEOL;
	    l3Log << "                            Quick Look Log File Path: " << sQLLogPath << dfmsEOL;
	    l3Log << "                                Mode Table File Path: " << sModeIDTablePath << dfmsEOL;
	    l3Log << "                                  pix0 fit File Path: " << sx0FitsDataPath << dfmsEOL;
	    l3Log << "                                  Gnu plot File Path: " << sPlotFilePath << dfmsEOL;
	    l3Log << "                             GnuPlot Executable Path: " << sGNUplotPath << dfmsEOL;
	    l3Log << "                                Mode Table File Name: " << sModeIDTableFilename << dfmsEOL;
	    l3Log << "                          Pixel Gain Table Base Name: " << sPixelGainTableBaseName << dfmsEOL;
	    l3Log << "                                    Instrument Model: " << sInstrumentModel << dfmsEOL;
	    l3Log << "                                Gain Table Base Name: " << sGainTableBaseName << dfmsEOL;
	    l3Log << "                         Number of Pixel Gain Tables: " << NUMPG << dfmsEOL;
	    l3Log << "       Range over which to search for GCU calib mass: " << GCUMASSDEL << dfmsEOL;
	    l3Log << "    Range over which to search for nonGCU verif mass: " << NONGCUMASSDEL << dfmsEOL;
	    l3Log << "         Flag to check for L3 Info Log file creation: " << L3INFOTOFILE << dfmsEOL;
	    l3Log << "          Flag for Quick Look Info Log file creation: " << QLINFOFILE << dfmsEOL;
	    l3Log << "         Flag for Diagnostic Plot Data file creation: " << DIAGTOFILE << dfmsEOL;
	    l3Log << " The first pixel to consider for mass peak discovery: " << PKLOWPT << dfmsEOL;
	    l3Log << "  The last pixel to consider for mass peak discovery: " << PKHIGHPT << dfmsEOL;
	    l3Log << "                                GCU Low Mass cut off: " << GCULOWCUTOFF << dfmsEOL;
	    l3Log << "                               GCU High Mass cut off: " << GCUHICUTOFF << dfmsEOL;
	    l3Log << "                                SLF Low Mass cut off: " << SLFLOWCUTOFF << dfmsEOL;
	    l3Log << "                               SLF High Mass cut off: " << SLFHICUTOFF << dfmsEOL;
	    l3Log << "   PPM diff used to compare target mass to peak mass: " << PPMDIFFMIN << dfmsEOL;
	    l3Log << "          Enhanced Noise Threshold level in ions/sec: " << ENOISET << dfmsEOL;
	    l3Log << "                        Standard Cal Deviation Value: " << STDCALDEV << dfmsEOL;
	    l3Log << "   Minimum counts above Mean Offset required for fit: " << PKHEIGHTLIMIT << dfmsEOL;
	    l3Log << "   Min number of GCU m0 vs pix0 pairs needed for fit: " << MIN4GCUFIT << dfmsEOL;
	    l3Log << "The maximum diff in minutes required for COPS search: " << COPSDELTA << dfmsEOL;
	    l3Log << " The # of Std Dev above meanOffset to look for peaks: " << NUMSDEV << dfmsEOL;
	    l3Log << "   Max # of peaks to store during prelim peak search: " << MAXNUMPEAKS << dfmsEOL;
	    l3Log << "                                       Bad data Flag: " << BADDATA << dfmsEOL;
	    l3Log << "     Flag to force conversion of out of bounds peaks: " << CONVERTOUTBPEAK << dfmsEOL;
	    l3Log << "    Flag to force conversion of out of low Amp peaks: " << CONVERTLOWAMP << dfmsEOL;
	    l3Log << "          Flag to force conversion of High Res modes: " << CONVERTHIGHRES << dfmsEOL;
	    l3Log << "----------------------------------------------------------------------------" << dfmsEOL;
	}
  
	toLog << dfmsEOL;
    toLog << "----------------------------------------------------------------------------" << dfmsEOL;
    toLog << "   ----- Begin Logging current L2 to L3 process at: "<< getProcessTime(2) << "------" << dfmsEOL;
    toLog << "----------------------------------------------------------------------------" << dfmsEOL;
    toLog << "                      Inputs from Configuration file: " << cfgFile << dfmsEOL;
    toLog << "                     Inputs DFMS Constants from file: " << constantsFile << dfmsEOL;
    toLog << "                     DFMS L2 to L3 install directory: " << sInstallDir << dfmsEOL;
    toLog << "                               L3 Info Log File Path: " << sL3InfoLogPath << dfmsEOL;
    toLog << "                                       Log File Path: " << sLogPath << dfmsEOL;
    toLog << "                                      L2 Source Path: " << sL2sourcePath << dfmsEOL;
    toLog << "                                      L3 Output Path: " << sL3outputPath << dfmsEOL;
    toLog << "                                      MPS Table Path: " << sMassPeakTablePath << dfmsEOL;
    toLog << "                                     Gain Table Path: " << sGainTablePath << dfmsEOL;
    toLog << "                                      COPS Data Path: " << sCOPSdataPath << dfmsEOL;
    toLog << "                                      Plot File Path: " << sPlotFilePath << dfmsEOL;
    toLog << "                           Diagnostic Plot Data Path: " << sDiagPath << dfmsEOL;
    toLog << "                            Quick Look Log File Path: " << sQLLogPath << dfmsEOL;
    toLog << "                                Mode Table File Path: " << sModeIDTablePath << dfmsEOL;
    toLog << "                                  pix0 fit File Path: " << sx0FitsDataPath << dfmsEOL;
    toLog << "                                  Gnu plot File Path: " << sPlotFilePath << dfmsEOL;
    toLog << "                             GnuPlot Executable Path: " << sGNUplotPath << dfmsEOL;
    toLog << "                                Mode Table File Name: " << sModeIDTableFilename << dfmsEOL;
    toLog << "                          Pixel Gain Table Base Name: " << sPixelGainTableBaseName << dfmsEOL;
    toLog << "                                    Instrument Model: " << sInstrumentModel << dfmsEOL;
    toLog << "                                Gain Table Base Name: " << sGainTableBaseName << dfmsEOL;
    toLog << "                         Number of Pixel Gain Tables: " << NUMPG << dfmsEOL;
    toLog << "       Range over which to search for GCU calib mass: " << GCUMASSDEL << dfmsEOL;
    toLog << "    Range over which to search for nonGCU verif mass: " << NONGCUMASSDEL << dfmsEOL;
    toLog << "         Flag to check for L3 Info Log file creation: " << L3INFOTOFILE << dfmsEOL;
    toLog << "          Flag for Quick Look Info Log file creation: " << QLINFOFILE << dfmsEOL;
    toLog << "         Flag for Diagnostic Plot Data file creation: " << DIAGTOFILE << dfmsEOL;
    toLog << " The first pixel to consider for mass peak discovery: " << PKLOWPT << dfmsEOL;
    toLog << "  The last pixel to consider for mass peak discovery: " << PKHIGHPT << dfmsEOL;
    toLog << "                                GCU Low Mass cut off: " << GCULOWCUTOFF << dfmsEOL;
    toLog << "                               GCU High Mass cut off: " << GCUHICUTOFF << dfmsEOL;
    toLog << "                                SLF Low Mass cut off: " << SLFLOWCUTOFF << dfmsEOL;
    toLog << "                               SLF High Mass cut off: " << SLFHICUTOFF << dfmsEOL;
    toLog << "   PPM diff used to compare target mass to peak mass: " << PPMDIFFMIN << dfmsEOL;
    toLog << "          Enhanced Noise Threshold level in ions/sec: " << ENOISET << dfmsEOL;
    toLog << "                        Standard Cal Deviation Value: " << STDCALDEV << dfmsEOL;
    toLog << "   Minimum counts above Mean Offset required for fit: " << PKHEIGHTLIMIT << dfmsEOL;
    toLog << "   Min number of GCU m0 vs pix0 pairs needed for fit: " << MIN4GCUFIT << dfmsEOL;
    toLog << "The maximum diff in minutes required for COPS search: " << COPSDELTA << dfmsEOL;
    toLog << " The # of Std Dev above meanOffset to look for peaks: " << NUMSDEV << dfmsEOL;
    toLog << "   Max # of peaks to store during prelim peak search: " << MAXNUMPEAKS << dfmsEOL;
    toLog << "                                       Bad data Flag: " << BADDATA << dfmsEOL;
    toLog << "     Flag to force conversion of out of bounds peaks: " << CONVERTOUTBPEAK << dfmsEOL;
    toLog << "    Flag to force conversion of out of low Amp peaks: " << CONVERTLOWAMP << dfmsEOL;
    toLog << "          Flag to force conversion of High Res modes: " << CONVERTHIGHRES << dfmsEOL;
    toLog << "----------------------------------------------------------------------------" << dfmsEOL;


	//  ---------------- Get required Tables: ModeID, Pixel Gain,... Data ---------------------

    // Step 4 - Create the Mode ID Table object and read the Mode ID Table data
    sModeIDTableFilename = DFMS_ModeID_Table_Class::findLatestModeTable(sModeIDTablePath);
    modeIDobject = new DFMS_ModeID_Table_Class(sModeIDTablePath, sModeIDTableFilename);
    nSuccess = modeIDobject->getModeData();
    if (!nSuccess) {
		sErrorMessage = "Unable to get Mode ID Table file: "+file;
		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
		cout << "Unable to get Mode ID Table file: " << file << endl;
		exit(EXIT_FAILURE);
	} else {
		sInfoMessage = "Successfully found: "+sModeIDTableFilename;
		writeToLog(sInfoMessage, sFunctionName, INFO);
		cout << sInfoMessage << endl;
	}
    if (verbose >= 2) modeIDobject->printModeFile("All");
     
    // If we're are not in CEM mode then get the Gain/PixelGain/MPST tables
    if (!DOCEM) {
        
    	// Step 5 - Get All Gain Table file info from directory
        gtPath = sGainTablePath+sInstrumentModel+"/";
        nSuccess = DFMS_Gain_Table_Class::getGTFileInfo(gtPath,GTFileInfo);
        if (!nSuccess) {
            sErrorMessage = "Unable to get Gain Table dir info at: ";
            sErrorMessage += sGainTablePath;
            writeToLog(sErrorMessage, sFunctionName, ERROR);
            cout << "Unable to get Gain Table directory info at: " << endl;
            exit(EXIT_FAILURE);
        } else {
            sInfoMessage = "Successfully found: "+util_intToString(GTFileInfo.size());
            sInfoMessage += " Gain Table Files(s)";
            writeToLog(sInfoMessage, sFunctionName, INFO);
            cout << sInfoMessage << endl;
            for (int i=0; i<GTFileInfo.size(); i++) {
                cout << gtPath+GTFileInfo[i].filename << endl;
                toLog << gtPath+GTFileInfo[i].filename << dfmsEOL;
                if (L3INFOTOFILE) l3Log << gtPath+GTFileInfo[i].filename << dfmsEOL;
            }
        }
        // STEP 5a - Read all Gain Tables. Store relevant info in the vector structure GTFileInfo
        gtObject = new DFMS_GainTable_Interp_Class(gtPath);
        nSuccess = gtObject->readAllGT(GTFileInfo);
    	if (!nSuccess) {
    		sErrorMessage = "Unable to get Gain Table file: "+file;
    		writeToLog(sErrorMessage, sFunctionName, ERROR);
    		cout << "Unable to get Gain Table file: " << file << endl;
    		exit(EXIT_FAILURE);
    	}
        
    	// Step 6 - Get All Pixel Gain Table file info from directory
    	pgtPath = sGainTablePath+sInstrumentModel+"/";
    	nSuccess = DFMS_PixelGain_Table_Class::getPGTFileInfo(pgtPath, PGTFileInfo);
    	if (!nSuccess) {
    		sErrorMessage = "Unable to get Pixel Gain Table dir info at: ";
    		sErrorMessage += sGainTablePath;
    		writeToLog(sErrorMessage, sFunctionName, ERROR);
    		cout << "Unable to get Pixel Gain Table directory info at: " << endl;
    		exit(EXIT_FAILURE);
    	} else {
    		sInfoMessage = "Successfully found: "+util_intToString(PGTFileInfo.size());
    		sInfoMessage += " Pixel Gain Table Files(s)";
    		writeToLog(sInfoMessage, sFunctionName, INFO);
    		cout << sInfoMessage << endl;
    		for (int i=0; i<PGTFileInfo.size(); i++) {
    			cout << pgtPath+PGTFileInfo[i].dirName << " GainStep = " << PGTFileInfo[i].GainStep  << endl;
    			toLog << pgtPath+PGTFileInfo[i].dirName << " GainStep = " << PGTFileInfo[i].GainStep  << dfmsEOL;
    			if (L3INFOTOFILE) l3Log << pgtPath+PGTFileInfo[i].dirName << " GainStep = " << PGTFileInfo[i].GainStep  << dfmsEOL;
    		}
    	}

    	// STEP 6a - Read all Pixel Gain Tables.  Store relevant info in the vector structure PGTFileInfo
    	pgObj = new DFMS_PixelGain_Interp_Class(pgtPath);
        cout << "PGTFileInfo size = " << PGTFileInfo.size() << endl;
    	pgObj->readAllPGT(PGTFileInfo);

    	// Step 7 - Get All GCU Mass Peak Search Table file info from directory
    	nSuccess = DFMS_MPST_Class::getMPSTfilesInDir(sMassPeakTablePath, gcuMPSTinfo, "GCU");
    	if (!nSuccess) {
    		sErrorMessage = "Unable to get GCU Mass Peak Search Table dir info at: ";
    		sErrorMessage += sMassPeakTablePath;
    		writeToLog(sErrorMessage, sFunctionName, ERROR);
    		cout << "Unable to get GCU Mass Peak Search Table directory info at: " << sMassPeakTablePath << endl;
    		exit(EXIT_FAILURE);
    	} else {
    		sInfoMessage = "Successfully found: "+util_intToString(gcuMPSTinfo.size());
    		sInfoMessage += " GCU MPST file(s)";
    		writeToLog(sInfoMessage, sFunctionName, INFO);
    		cout << sInfoMessage << endl;
    		for (int i=0; i<gcuMPSTinfo.size(); i++) {cout << "file: " << gcuMPSTinfo[i].fileName << endl;}
    	}

    	// Step 8 - Get All nonGCU Mass Peak Search Table file info from directory
    	nSuccess = DFMS_MPST_Class::getMPSTfilesInDir(sMassPeakTablePath, nonGCUMPSTinfo, "nonGCU");
    	if (!nSuccess) {
    		sErrorMessage = "Unable to get nonGCU Mass Peak Search Table dir info at: ";
    		sErrorMessage += sMassPeakTablePath;
    		writeToLog(sErrorMessage, sFunctionName, ERROR);
    		cout << "Unable to get nonGCU Mass Peak Search Table directory info at: " << sMassPeakTablePath << endl;
    		exit(EXIT_FAILURE);
    	} else {
    		sInfoMessage = "Successfully found: "+util_intToString(nonGCUMPSTinfo.size());
    		sInfoMessage += " SLF MPST file(s)";
    		writeToLog(sInfoMessage, sFunctionName, INFO);
    		cout << sInfoMessage << endl;
    		for (int i=0; i<nonGCUMPSTinfo.size(); i++) {cout << "file: " << nonGCUMPSTinfo[i].fileName << endl;}
    	}
    }
	//  ------------------------------ Now get the L2 File(s) ------------------------------

    // Create an object that examines a directory full of L2 files.  Store the pertinent info.
    // Assumes files are in sL2sourcePath.
    //
	L2DirObj = new DFMS_PDS_L2_dir_Class(sL2sourcePath);

	// Step 9 - Get all L2 files in the source (sL2sourcePath) directory
	// Note:  To run one file use -f command line argument or simply put the single file into a
	// directory and ensure that you use that directory name as the sL2sourcePath
    // in your configuration file.
	L2DirObj->getL2FilesInDir(intMode);

	if (verbose >=3) {
		for (int i=0; i<L2DirObj->L2Info.size(); i++) {
			L2DirObj->printFileInfo(i);
		}
	}

	// Total L2 files found
    numL2Files = L2DirObj->getNumFiles();

    // Check if in CEM/MCP configuration is correct
    if (DOCEM) {

		if (L2DirObj->L2Info[0].fileName.substr(0,2).compare("MC") == 0) {
			cout << "\n\n----------------------------------------------------------------------------\n";
			cout << "The variable DOCEM is set to \"true\" but you are processing MCP files !?!?!\n";
			cout << "Did you forget to change DOCEM to \"false\" in ConfigFile.dat?\n";
			cout << "----------------------------------------------------------------------------\n\n";
			toLog << "\n\n----------------------------------------------------------------------------\n";
			toLog << "The variable DOCEM is set to \"true\" but you are processing MCP files !?!?!\n";
			toLog << "Did you forget to change DOCEM to \"false\" in ConfigFile.dat?\n";
			toLog << "----------------------------------------------------------------------------\n\n";
			exit(EXIT_FAILURE);
		}

    } else {

    	if (L2DirObj->L2Info[0].fileName.substr(0,2).compare("CE") == 0) {
    		cout << "\n\n----------------------------------------------------------------------------\n";
    		cout << "The variable DOCEM is set to \"false\" but you are processing CEM files !?!?!\n";
    		cout << "Did you forget to change DOCEM to \"true\" in ConfigFile.dat?\n";
			cout << "----------------------------------------------------------------------------\n\n";
    		toLog << "\n\n----------------------------------------------------------------------------\n";
    		toLog << "The variable DOCEM is set to \"false\" but you are processing CEM files !?!?!\n";
    		toLog << "Did you forget to change DOCEM to \"true\" in ConfigFile.dat?\n";
			toLog << "----------------------------------------------------------------------------\n\n";
    		exit(EXIT_FAILURE);
    	}
    }

    // Step 10 - Get COPS Data

    // First get COPS NG Data files info from NG directory
    nSuccess = DFMS_COPS_Class::getCOPSfilesInDir("ng", sCOPSdataPath, COPSNGinfo);
    if (!nSuccess) {
		sErrorMessage = "Unable to get COPS NG Data at: ";
        sErrorMessage += sCOPSdataPath + " or no data exists";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
        cout << sErrorMessage << endl;
	} else {
        sInfoMessage = "Successfully found: "+util_intToString(COPSNGinfo.size());
        sInfoMessage += " COPS NG files";
        writeToLog(sInfoMessage, sFunctionName, INFO);
    	cout << sInfoMessage << endl;
    }
     
    // Next get COPS RG Data files info from RG directory
    nSuccess = DFMS_COPS_Class::getCOPSfilesInDir("rg", sCOPSdataPath, COPSRGinfo);
    if (!nSuccess) {
		sErrorMessage = "Unable to get COPS RG Data at: ";
        sErrorMessage += sCOPSdataPath + " or no data exists";;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		cout << sErrorMessage << endl;
	} else {
        sInfoMessage = "Successfully found: "+util_intToString(COPSRGinfo.size());
        sInfoMessage +=" COPS RG files";
        writeToLog(sInfoMessage, sFunctionName, INFO);
    	cout << sInfoMessage << endl;
    }

    // Lastly get COPS BG Data files info from BG directory
    nSuccess = DFMS_COPS_Class::getCOPSfilesInDir("bg", sCOPSdataPath, COPSBGinfo);
    if (!nSuccess) {
    	sErrorMessage = "Unable to get COPS BG Data info at: ";
    	sErrorMessage += sCOPSdataPath + " or no data exists";;
    	writeToLog(sErrorMessage, sFunctionName, ERROR);
    	cout << sErrorMessage << endl;
    } else {
    	sInfoMessage = "Successfully found: "+util_intToString(COPSBGinfo.size());
    	sInfoMessage +=" COPS BG files";
    	writeToLog(sInfoMessage, sFunctionName, INFO);
    	cout << sInfoMessage << endl;
    }

    // Check if in CEM or MCP mode.  If in MCP then check if Phase I fitting to create X0 fit
    // files is required
    if (DOCEM) {

		if (doOneFile) {
			begId = L2DirObj->getFileIndex(dfmsFileName);
			endId = begId+1;
		} else {
			begId = 0;
			endId = numL2Files;
		}

    } else {

    	// Check if only one L2 file is requested
    	if (doOneFile) {
    		begId = L2DirObj->getFileIndex(dfmsFileName);
    		endId = begId+1;

    	} else {   // Do all files

    		begId = 0;
    		endId = numL2Files;

    		// The following flag is set to true in the Config file so that the pix0 vs m0 fitting
    		// section of the code is skipped if X0Fit files have already been created in
    		// an earlier run (for this specific set of L2 files)
    		if (processNewGCUSLFfiles) {

    			// PHASE I - Go through all GCU and SLF files and proceed to create needed
    			// pix0 fit files.  Note that GCU files will only be created if they exit
    			// in the current batch of L2 files.  Regardless GCU files are required to exist
    			// in order for the SLF files to be properly handled.  Therefore if no new GCU
    			// files exist then the code looks for the latest GCU files.  If none are found
    			// then a fatal error occurs and the code exits. In this loop GCU files
    			// (when they exist) are handled first and then SLF files.  NonGCUnonSLF
    			// files are skipped here and  handled in Phase II below.

    			// create L3Info temporarily.
    			L3Info = new DFMS_L3Info_Class();

    			// STEP 11 - L2DirObj->processPix0FitData Loops through begId->endId L2
    			// files initially to create GCU and Verification peak (SLF) m0 vs pix0 fit files.
    			int numPix0created=0;
    			nSuccess = L2DirObj->processPix0FitData(begId, endId, gtObject, GTFileInfo, modeIDobject,
    				               pgObj, PGTFileInfo, gcuMPSTinfo, nonGCUMPSTinfo, &numPix0created);
    			if (!nSuccess) {
    				sErrorMessage = "Unable to process the L2 files to create pix0 fit files ";
    				writeToLog(sErrorMessage, sFunctionName, WARN);
    				cout << " - Warning - " + sErrorMessage << endl;
    			} else {
    				sInfoMessage = "Created  "+util_intToString(numPix0created)+" Pix0 fit files";
    				writeToLog(sInfoMessage, sFunctionName, INFO);
    				cout << sInfoMessage << endl;
    			}

    			// Remove the L2 files which failed the fitting portion of processPix0FitData above.
    			// These will be removed from the L2Info vector.  They will not be processed in
    			// the L2->L3 conversion section that follows.

    			delete L3Info, L3Info=0;

    	    	// Dump the results of the L2 files identified whose pix0 points will be used
    	    	// in GCU and SLF pix0 vs m0 Fit files.
    	    	cout << "\n---------------------------------------------------" << endl;
	    		cout << "  ---- Result of developing m0 vs pix0 fits ------" << endl;
    	    	cout << "  ---- Initial pix0 vs m0 Points Identified ------" << endl;
    	   		cout << "-----------------------------------------------------" << endl;
    	    	cout << "GCU LowMass  LowRes  Row A/B Points = " << gcuLmLrGoodFit << endl;
    	    	cout << "GCU HighMass LowRes  Row A/B Points = " << gcuHmLrGoodFit << endl;
    	    	cout << "GCU LowMass  HighRes Row A/B Points = " << gcuLmHrGoodFit << endl;
    	    	cout << "GCU MedMass  HighRes Row A/B Points = " << gcuMmHrGoodFit << endl;
    	    	cout << "GCU HighMass HighRes Row A/B Points = " << gcuHmHrGoodFit << endl;
    	    	cout << "SLF LowMass  LowRes  Row A/B Points = " << slfLmLrGoodFit << endl;
    	    	cout << "SLF HighMass LowRes  Row A/B Points = " << slfHmLrGoodFit << endl;
    	    	cout << "SLF LowMass  HighRes Row A/B Points = " << slfLmHrGoodFit << endl;
    	    	cout << "SLF MedMass  HighRes Row A/B Points = " << slfMmHrGoodFit << endl;
    	    	cout << "SLF HighMass HighRes Row A/B Points = " << slfHmHrGoodFit << endl;
    	    	cout << "-----------------------------------------------------" << endl << endl;

    	    	if (L3INFOTOFILE) {
    	    		l3Log << dfmsEOL;
    	    		l3Log << "-----------------------------------------------------" << dfmsEOL;
    	    		l3Log << "  ---- Result of developing m0 vs pix0 fits ------" << dfmsEOL;
    	    		l3Log << "  ---- Initial pix0 vs m0 Points Identified ------" << dfmsEOL;
    	    		l3Log << "-----------------------------------------------------" << dfmsEOL;
    	    		l3Log << "GCU LowMass  LowRes  Row A/B Points = " << gcuLmLrGoodFit << dfmsEOL;
    	    		l3Log << "GCU HighMass LowRes  Row A/B Points = " << gcuHmLrGoodFit << dfmsEOL;
    	    		l3Log << "GCU LowMass  HighRes Row A/B Points = " << gcuLmHrGoodFit << dfmsEOL;
    	    		l3Log << "GCU MedMass  HighRes Row A/B Points = " << gcuMmHrGoodFit << dfmsEOL;
    	    		l3Log << "GCU HighMass HighRes Row A/B Points = " << gcuHmHrGoodFit << dfmsEOL;
    	    		l3Log << "SLF LowMass  LowRes  Row A/B Points = " << slfLmLrGoodFit << dfmsEOL;
    	    		l3Log << "SLF HighMass LowRes  Row A/B Points = " << slfHmLrGoodFit << dfmsEOL;
    	    		l3Log << "SLF LowMass  HighRes Row A/B Points = " << slfLmHrGoodFit << dfmsEOL;
    	    		l3Log << "SLF MedMass  HighRes Row A/B Points = " << slfMmHrGoodFit << dfmsEOL;
    	    		l3Log << "SLF HighMass HighRes Row A/B Points = " << slfHmHrGoodFit << dfmsEOL;
    	    		l3Log << "-----------------------------------------------------" << dfmsEOL << dfmsEOL;
    	    	}
    		}
    	}
    }

    //begId = 227;    // These two lines are used for debugging purposes.
    //endId = 228;

    // PHASE II:  Using the results from PHASE I above or appropriate x0Fit files already available
    // proceed to mass scale calibration, ppmDiff calculation, and calculation of all information
    // needed for creating PDS compliant L3 output files.


    // STEP 12 - Now Loop through all L2 files again and if MCP then complete the L2->L3 conversion process
    // by using either the GCU/SLF pix0 fit results from the FIRST LOOP or using pre-existing pix0 fit results.
    // Also calculate the relevant PPM Diff, subsequent Quality ID flags, and finally writing L3 PDS science
    // data with mass Scale and Ions counted.  This loop repeats the general process of the first loop in
    // processPix0FitData (Step 11) above but completes the details in order to write the L3 PDS data to file
    for (int i=begId; i<endId; i++) {

    	// STEP 12.1 - Get Mode Info
    	if (modeData.count(L2DirObj->L2Info[i].mode) <= 0) {
        	sErrorMessage = "Mode: "+util_intToString(L2DirObj->L2Info[i].mode);
        	sErrorMessage += " is not recognized - Skipping file processing";
        	writeToLog(sErrorMessage, sFunctionName, ERROR);
        	totFilesSkipped++;
        	cout << sErrorMessage << endl;
        	continue;
    	} else {
    		modeInfo = modeData[L2DirObj->L2Info[i].mode];
    	    //modeInfo.printData();
    	}

        // Set emission value for this L2 file mode for Time Series calculations
        emission = modeInfo.emission;

        // STEP 12.2 - Define the current L2 file name
        L2File = L2DirObj->L2Info[i].fileName;
        //cout << endl<< "Next L2 file to process: " << L2File << endl;

        // Get the file date in Julian seconds to be used for various checks
        double thisFileDate = rosToSec(L2DirObj->L2Info[i].dateTime, 1);

        // Check if this L2 file date is in exclusion zone.  Files with dates within this
        // zone will NOT be processed for various reasons.
        bool exclude = dateException(thisFileDate);
        if (exclude) {
        	skipReas["fileExcluded"]++;
        	sInfoMessage = "File is within date/time exclusion region: "+L2File;
        	writeToLog(sInfoMessage, sFunctionName, INFO);
        	totFilesSkipped++;
        	if (QLINFOFILE) {
        		QuickLookLog << " - Skipped - File is within date/time exclusion region: " << L2File << dfmsEOL;
        	    qlCount++;
        	    if (qlCount % QLPPAGE == 0) qlHeader();
        	}
        	cout << " - Skipped - File is within date/time exclusion region: " << L2File << endl;
        	continue;
        }

		// Check for New PDS Header time boundary. New L2 PDS Header is assumed for
		// all times after 01/01/2014 00:00:00
		string sJan012014 = "20140101_000000000";
		double jan012014 = rosToSec(sJan012014, 1);
		if (thisFileDate < jan012014) {
			isNewPDS = false;
		} else {
			isNewPDS = true;
		}

        // If skipping phase I processing then we need to define
        // GCU existence time flag isRecentGCUavailable
        if (!processNewGCUSLFfiles && !DOCEM) {
        	string sJan032015 = "20150103_000000000";
        	double jan032015 = rosToSec(sJan032015, 1);

        	// Check if we are in the time domain before or after the GCU failure
        	if (thisFileDate < jan032015) {
        		isRecentGCUavailable = true;
        	} else {
        		isRecentGCUavailable = false;
        	}

        }

        // Write to quick look file (if requested)
        if (QLINFOFILE) {
        	char qcline[200];
        	string qsline;
        	sprintf(qcline,"%4.4d  %s",i,L2File.c_str());
        	qsline.assign(qcline);
        	QuickLookLog << qsline << "  ";
        }

        // Write to file index to screen and L3InfoLog (if requested)
        cout << endl << endl;
        cout << "--------------------- Trying (Main): " << i << " of " << endId-1 << " - L2 File: " << L2File;
        cout << " ---------------------" << endl;
        if (L3INFOTOFILE) {
        	l3Log << dfmsEOL << dfmsEOL;
        	l3Log << "--------------------- Trying: " << i << " of " << endId-1 << " - L2 File: " << L2File;
        	l3Log << " ---------------------" << dfmsEOL;
        }

        // Track mode statistic
        modesPresent[L2DirObj->L2Info[i].mode]++;

        // STEP 12.3 - Create the current L3Info Object. The data in this Class will be filled in
        // as the L2->L3 process proceeds.  Later it is used to populate the required L3 PDS Header
        // and L3 HouseKeeping items of the L3 PDS file.
        if (!DOCEM) {
        	L3Info = new DFMS_L3Info_Class();
        	// Set some needed L3Info data if in MCP mode
        	L3Info->Pix0Calc = true;
        	L3Info->L2baseName = L2File.substr(0,27);
        	L3Info->isGCU = L2DirObj->L2Info[i].isGCU;
        }

       	sInfoMessage = "--------------- "+util_intToString(i)+" - Processing L2 File: "+L2File;
        writeToLog(sInfoMessage, sFunctionName, INFO);

        // Increment the statistic counting attempted L2 files
        totFilesAttempted++;

        // If MCP file then proceed to Step 12.4 otherwise skip Step 12.4,12.5
        if (!DOCEM) {
        	// STEP 12.4 - Create the L2File object and read its data
        	L2Obj = new DFMS_PDS_L2_file_Class(sL2sourcePath,L2File);
        	nSuccess = L2Obj->getL2file(L2DirObj->L2Info[i]);
        	if (!nSuccess) {
        		sErrorMessage = "Unable to open/read/form L2 file: "+L2File;
        		sErrorMessage += " - Skipping file processing";
        		writeToLog(sErrorMessage, sFunctionName, ERROR);
        		delete L2Obj; L2Obj=0;

        		delete L3Info, L3Info=0;
        		totFilesSkipped++;
        		cout << " - Skipped - Unable to open/read/form proper L2 file" << endl;
        		continue;
        	}
        	if (verbose >= 2) L2Obj->printL2File("All");

        	// STEP 12.5 - Check if this is a high res mode.  If so warn that low res massscale is being used
        	// and proceed.  Skip (if requested)
        	if (L2DirObj->L2Info[i].hiRes) {
        		if (CONVERTHIGHRES) {
        			sInfoMessage = "Alert: This is a High res mode but using Low res mass Scale";
        			writeToLog(sErrorMessage, sFunctionName, WARN);
        			L3Info->Pix0Calc = false;
        			exceptionType["highRes"]++;
        		} else {
        			skipReas["highRes"]++;
        			sErrorMessage = "Currently not processing High Resolution files: ";
        			sErrorMessage += L2File +" - Skipping file processing";
        			writeToLog(sErrorMessage, sFunctionName, WARN);
        			delete L2Obj; L2Obj=0;

        			delete L3Info, L3Info=0;
        			totFilesSkipped++;
        			if (QLINFOFILE) {
        				QuickLookLog << " - Skipped - Currently not processing High Resolution files" << dfmsEOL;
        				qlCount++;
        				if (qlCount % QLPPAGE == 0) qlHeader();
        			}
        			continue;
        		}
        	}
        }

        // STEP 12.6 - Only handle MCP and CEM data
        if ((modeInfo.detector.compare("MCP") != 0) && (modeInfo.detector.compare("CEM") != 0)) {
        	skipReas["notMCPorCEM"]++;
        	sErrorMessage = "At present only DFMS MCP and CEM processing is supported ";
        	sErrorMessage += "Current mode is:"+modeInfo.detector;
        	sErrorMessage += " - Skipping file processing";
        	writeToLog(sErrorMessage, sFunctionName, WARN);
        	delete L2Obj; L2Obj=0;
        	if (L3Info != NULL) {
        		delete L3Info, L3Info=0;
        	}
        	totFilesSkipped++;
        	if (QLINFOFILE) {
        		QuickLookLog << " - Skipped - At present only DFMS MCP and CEM processing is supported" << dfmsEOL;
        		qlCount++;
        		if (qlCount % QLPPAGE == 0) qlHeader();
        	}
        	cout << " - Skipped - At present only DFMS MCP and CEM processing is supported" << endl;
        	continue;
        }

        // STEP 12.8 - Get the bes COPS NG and RG, or BG Data files - sets name to global variables,
        // sBestNGCOPS, sBestRGCOPS, and sBestBGCOPS
        nSuccess = DFMS_COPS_Class::findBestCOPSfile(sCOPSdataPath,
        		                           L2DirObj->L2Info[i].dateTime,
                                           COPSNGinfo, COPSRGinfo, COPSBGinfo);
        if (!nSuccess) {
             sErrorMessage = "Unable to find an appropriate COPS NG/RG/BG file: ";
             sErrorMessage += sBestNGCOPS+" -/- "+sBestRGCOPS+" -/- "+sBestBGCOPS;
             writeToLog(sErrorMessage, sFunctionName, ERROR);
        } else {
             sInfoMessage = "Best COPS NG/RG/BG files found are: ";
             sInfoMessage += sBestNGCOPS+" -/- "+sBestRGCOPS+" -/- "+sBestBGCOPS;
             writeToLog(sInfoMessage, sFunctionName, INFO);
             cout << sInfoMessage << endl;
        }

    	// Create the L3 (MCP or CEM) filename
    	L3File = L2File.substr(0,18)+"_3_"+L2File.substr(22,5)+".TAB";

    	// STEP 12.7 - If not CEM then Get Best GCU or nonGCU Mass Peak Table file -
        // sets name to global variable, sBestMPST
        if (!DOCEM) {
        	L3Info->L3FileName = L3File;
        	// Find Best MPST
        	if (L2DirObj->L2Info[i].isGCU) {
        		nSuccess = DFMS_MPST_Class::findBestMPSTfile(sMassPeakTablePath,
        				L2DirObj->L2Info[i].dateTime,
						L2DirObj->L2Info[i].isGCU, gcuMPSTinfo);
        	} else {
        		nSuccess = DFMS_MPST_Class::findBestMPSTfile(sMassPeakTablePath,
        				L2DirObj->L2Info[i].dateTime,
						L2DirObj->L2Info[i].isGCU, nonGCUMPSTinfo);
        	}
        	if (!nSuccess) {
        		skipReas["noMPSTfile"]++;
        		sErrorMessage = "Unable to find an appropriate MPST file: ";
        		sErrorMessage += sBestMPST+dfmsEOL;
        		writeToLog(sErrorMessage, sFunctionName, ERROR);
        		delete L2Obj; L2Obj=0;
        		delete L3Info; L3Info=0;
        		totFilesSkipped++;
        		if (QLINFOFILE) {
        			QuickLookLog << " - Skipped - Unable to find an appropriate MPST file" << dfmsEOL;
        			qlCount++;
        			if (qlCount % QLPPAGE == 0) qlHeader();
        		}
        		cout << " - Skipped - Unable to find an appropriate MPST file" << endl;
        		continue;
        	} else {
        		sInfoMessage = "Best MPST file found: "+sBestMPST;
        		writeToLog(sInfoMessage, sFunctionName, INFO);
        		cout << sInfoMessage << endl;
        	}
          
        	// STEP 12.9 - Get Best Pixel Gain Table.  Here we use the following algorithm.
        	// If the GS is shared by two or more Pixel gain tables then perform a linear interpolation.
        	// If there is only one Pixel Gain table for this GS
        	// then use the nearest PG Table in time and GS.

        	DFMS_PixelGain_Table_Class *pixGTObject = NULL;
        	int GSin = L2DirObj->L2Info[i].gainStep;
        	// Check how many PG Tables exist for this GS and store info into local variable PGSin.
        	// PGSin is a subset of PGTFileInfo corresponding to all PGTFileInfo members with the same
        	// gain step GS
        	vector<pgFileInfo> PGSin;
        	int mGS = pgObj->findMulti_GS_PGT(GSin, PGTFileInfo, PGSin);
        	// For PGSin membership of 1 or less use closest time/GS Pixel gain
        	if (mGS <= 1) {
        		int GSuse = 0;
        		string dType = "";
                string subDNme = "";
        		nSuccess = DFMS_PixelGain_Table_Class::findBestPGTtime(pgtPath, L2DirObj->L2Info[i].dateTime,
        							 	 	 	 	 	       GSin, &GSuse, dType, subDNme, PGTFileInfo);
        		if (!nSuccess) {
        			sErrorMessage = "Unable to find Best Pixel Gain Table Time ";
        			writeToLog(sErrorMessage, sFunctionName, ERROR);
        			cout << "Unable to find Best Pixel Gain Table Time " << endl;
        			exit(EXIT_FAILURE);
        		}

        		// STEP 12.10 - Get Pixel Gain Table info
        		char tmp[10];
        		sprintf(tmp,"%2.2d",GSuse);
        		string sGS = string(tmp);
        		string pgfile = sPixelGainTableBaseName+sBestPGTtime+"_"+dType+"_"+sInstrumentModel+"_GS"+sGS+".TAB";

        		// Create the Pixel Gain Table file path
        		string pgpath = pgtPath+subDNme+"/";
        		cout << "Using Pixel Gain File : " << pgfile << endl;
        		if (L3INFOTOFILE && verbose >= 3) l3Log << "Using Pixel Gain File : " << pgfile << dfmsEOL;

        		// Create the pixel Gain Table object then get the best pixel gain table file
        		// based on the nearest time and closest gain step
        		pixGTObject = new DFMS_PixelGain_Table_Class(pgpath, pgfile);
        		nSuccess = pixGTObject->getPGTable(pgpath, pgfile);
        		if (!nSuccess) {
        			sErrorMessage = "Reading Pixel Gain Table was unsuccessful ";
        			writeToLog(sErrorMessage, sFunctionName, ERROR);
        			cout << "Reading Pixel Gain Table was unsuccessful " << endl;
        			exit(EXIT_FAILURE);
        		}
        		// Optionally print Pixel Gain Table
        		if (verbose >= 2) pixGTObject->printpixGTtable("All");
        	} else {
        		// When PGSin contains 2 or more PG Tables for this GS use simple DFMS_PixelGain_Table_Class
        		// constructor.  This will create a simple PG Table object that will simply allow the usage
        		// of the pixGTdata field later in the processing.  Instead of reading pixel Gain data from
        		// file use the results of the interpolation to populate the pixGTdata array.
        		// Fill pixGTObject->pixGTdata with data from interpolation.
        		pixGTObject = new DFMS_PixelGain_Table_Class();
        		// use the PG Table Interp object pgObj to build the pixGTObject->pixGTdata array.
        		nSuccess = pgObj->buildPixGTdata(pixGTObject, GSin, thisFileDate, PGSin);
        		if (!nSuccess) {
        			// This should never happen but....
        			sErrorMessage = "Pixel Gain Interpolation was unsuccessful for unknown reason.";
        			writeToLog(sErrorMessage, sFunctionName, ERROR);
        			if (L3INFOTOFILE) l3Log << sErrorMessage << dfmsEOL;
        			cout << sErrorMessage << endl;
        			delete L2Obj; L2Obj=0;
        			delete L3Info, L3Info=0;
        			totFilesSkipped++;
        			cout << " - Skipped - Pixel Gain Interpolation was unsuccessful for unknown reason." << endl;
        			continue;
        		}
        	}

            // Now using the current L2 file date interpolate to find the correct Gain from the
            // available tables
            nSuccess = gtObject->buildGTdata(thisFileDate, GTFileInfo);
            if (!nSuccess) {
                // This should never happen but....
                sErrorMessage = "Gain Table Interpolation was unsuccessful for unknown reason.";
                writeToLog(sErrorMessage, sFunctionName, ERROR);
                if (L3INFOTOFILE) l3Log << sErrorMessage << dfmsEOL;
                cout << sErrorMessage << endl;
                delete L2Obj; L2Obj=0;
                delete L3Info, L3Info=0;
                totFilesSkipped++;
                cout << " - Skipped - Gain Table Interpolation was unsuccessful for unknown reason." << endl;
                continue;
            }
            
        	// STEP 12.11 - Get the overall gain for this L2 file
        	double gainA = gtObject->gtFit[GSin-1][2];
        	double gainB = gtObject->gtFit[GSin-1][3];
        	L2DirObj->L2Info[i].GS = GSin;

        	// Create the convert L2->L3 object.
        	mcpL2toL3 = new DFMS_ProcessMCP_Class(sMassPeakTablePath, sCOPSdataPath,
        				sL3outputPath, L3File, sx0FitsDataPath, gainA, gainB,
						pixGTObject->pixGTdata, L2Obj);

        	// STEP 12.12 - Process an L2 File into the Level 3 Science Data
        	nSuccess = mcpL2toL3->processL2(L2DirObj->L2Info[i],modeInfo);
        	if (!nSuccess) {
                // Increment commanded mass value fit failure
                thisRunCommMassFitFailed[L2DirObj->L2Info[i].commandedMass]++;
        		totFilesSkipped++;
        		sErrorMessage = "Conversion of MCP L2 file: "+L2File+" NOT successful";
        		writeToLog(sErrorMessage, sFunctionName, ERROR);
        		if (DIAGTOFILE) {
        			nSuccess = mcpL2toL3->writeL2L3DataFit(0, L2Obj->getFileName(),
        				   util_doubleToString(L2DirObj->L2Info[i].commandedMass),
        				   util_intToString(L2DirObj->L2Info[i].mode));
        			if (!nSuccess) {
        				sErrorMessage = "Unable to write MCP L2/L3 Data and Fit to file";
        				writeToLog(sErrorMessage, sFunctionName, WARN);
        			}
        		}
        	} else {
                // Increment commanded mass value successfully fit
                thisRunCommMassFitSuccess[L2DirObj->L2Info[i].commandedMass]++;
        		totFilesSuccessfullyProcessed++;
        		sInfoMessage = "Successfully converted file MCP L2: "+L2File+" to "+L3File;
        		writeToLog(sInfoMessage, sFunctionName, INFO);
        	}

        	// WARN that noo ppmdiff is possible for nonGCUnonSLF single file run
        	if (doOneFile && !L2DirObj->L2Info[i].isGCU && !L2DirObj->L2Info[i].isSLF) {
        		sInfoMessage = "Warning - PPMDiff is NOT DEFINED when running in single file mode";
        		writeToLog(sInfoMessage, sFunctionName, INFO);
        		cout << sInfoMessage << endl;
        		l3Log << sInfoMessage << dfmsEOL;
        	}

        	// STEP 12.13 - Release dynamically allocated memory of local variables
        	delete mcpL2toL3;  mcpL2toL3=0;
    		delete pixGTObject; pixGTObject=0;
            delete L2Obj;  L2Obj=0;
    		delete L3Info, L3Info=0;

        } else {  // Process a CEM file

			L2Obj = new DFMS_PDS_L2_file_Class(sL2sourcePath,L2File);
        	nSuccess = L2Obj->getL2file(L2DirObj->L2Info[i]);
        	if (!nSuccess) {
        		sErrorMessage = "Unable to open/read/form CEM L2 file: "+L2File;
        		sErrorMessage += " - Skipping file processing";
        		writeToLog(sErrorMessage, sFunctionName, ERROR);
        		delete L2Obj; L2Obj=0;
        		totFilesSkipped++;
        		cout << " - Skipped - Unable to open/read/form proper CEM L2 file" << endl;
        		continue;
        	}
        	if (verbose >= 2) L2Obj->printL2File("All");

        	// Create the CEML2toL3 object
        	cemL2toL3 = new DFMS_ProcessCEM_Class(sL3outputPath, L3File, sCOPSdataPath, L2Obj);

        	nSuccess = cemL2toL3->processL2(L2DirObj->L2Info[i]);
        	if (!nSuccess) {
        		totFilesSkipped++;
        		sErrorMessage = "Conversion of CEM L2 file: "+L2File+" NOT successful";
        		writeToLog(sErrorMessage, sFunctionName, ERROR);
        	} else {
        		totFilesSuccessfullyProcessed++;
        		sInfoMessage = "Successfully converted file CEM L2: "+L2File+" to "+L3File;
        		writeToLog(sInfoMessage, sFunctionName, INFO);
        	}

        	// Now build the CEM L3 file
        	nSuccess = cemL2toL3->createL3File();
        	if (!nSuccess) {
        		sErrorMessage = "Failed when writing L3 CEM file ";
        		writeToLog(sErrorMessage, sFunctionName, ERROR);
        	}

            delete cemL2toL3; cemL2toL3=0;
            delete L2Obj; L2Obj=0;

        }

	}

    // STEP 13 - Dump number of modes present stats for this session
    if (modesPresent.size() > 0) listModes();

    // STEP 14 - Dump pkPPMdiff stats to stdout and optionally to QuikLookLog/L3InfoLog files
	if (pkPPMvec.size() > 0) makePkPPMhisto();
    
    // STEP 14a - Dump Commanded Mass Statistics.  Number of Successfull Masses Processed
    //            and Unsuccessfull Masses Processed
    if (thisRunCommMassFitSuccess.size() > 0) listCommMassStats();

	// STEP 15 - Calculate time used for this run
    // Print the "Process Complete" message and close the Process Log -----------
    string timeAtStop = getProcessTime(4);
    int strt = util_stringToInt(timeAtStart);
    int stp = util_stringToInt(timeAtStop);
    double totTime = (double)(stp-strt)/CLOCKS_PER_SEC;

    // Get actual time in millisec at end of processing
    const long double sysTimeMS2 = time(0)*1000;
    const long double rTotTime = (sysTimeMS2-sysTimeMS1)/1000.0;

    // STEP 16 - Write out all end of run info to screen and pertinent log files

    // Write to process log file
   	toLog << dfmsEOL << dfmsEOL;
   	toLog << "----------------------------------------------------------------------" << dfmsEOL;
   	toLog << "        L2 Conversion Process Started at: " << sTstart << dfmsEOL;
   	toLog << "                Attempted L2 Conversions: " << totFilesAttempted << dfmsEOL;
   	toLog << "               Successful L2 conversions: " << totFilesSuccessfullyProcessed << dfmsEOL;
   	toLog << "---------------------------------------------------------------------- " << dfmsEOL;
   	toLog << "          High Resolution Mode Exception: " << exceptionType["highRes"] << dfmsEOL;
   	toLog << "    Peak Height Limit (Row  A) Exception: " << sideAexceptionType["peakHeightLimit"] << dfmsEOL;
   	toLog << "    Peak Height Limit (Row  B) Exception: " << sideBexceptionType["peakHeightLimit"] << dfmsEOL;
   	toLog << "   Peak Out of Bounds (Row  A) Exception: " << sideAexceptionType["peakOutOfBounds"] << dfmsEOL;
   	toLog << "   Peak Out of Bounds (Row  B) Exception: " << sideBexceptionType["peakOutOfBounds"] << dfmsEOL;
   	toLog << " No Useful peak found (Row  A) Exception: " << sideAexceptionType["peakFinderError"] << dfmsEOL;
   	toLog << " No Useful peak found (Row  B) Exception: " << sideBexceptionType["peakFinderError"] << dfmsEOL;
   	toLog << "---------------------------------------------------------------------- " << dfmsEOL;
   	toLog << "   No GCU MPST file found - File Skipped: " << skipReas["noMPSTfile"] << dfmsEOL;
   	toLog << "     GCU Mass Not in MPST - File Skipped: " << skipReas["gcuMassNotInMPST"] << dfmsEOL;
   	toLog << "   Not an MCP or CEM file - File Skipped: " << skipReas["notMCPorCEM"] << dfmsEOL;
   	toLog << "     High Resolution Mode - File Skipped: " << skipReas["highRes"] << dfmsEOL;
   	toLog << "   ROSINA HK Mass is Zero - File Skipped: " << skipReas["rosMassZero"] << dfmsEOL;
   	toLog << "        Peak Height Limit - File Skipped: " << skipReas["peakHeightLimit"] << dfmsEOL;
   	toLog << "         Gain Not Defined - File Skipped: " << skipReas["gainNotDef"] << dfmsEOL;
   	toLog << "       Pix0 out of Bounds - File Skipped: " << skipReas["pix0Bounds"] << dfmsEOL;
   	toLog << "        Peak Finder Error - File Skipped: " << skipReas["peakFinderError"] << dfmsEOL;
   	toLog << "  L2 HK with Line OverRun - File Skipped: " << skipReas["L2HKLineOverun"] << dfmsEOL;
   	toLog << "L2 file in Exclusion Zone - File Skipped: " << skipReas["fileExcluded"] << dfmsEOL;
   	toLog << "                  Total L2 Files Skipped: " << totFilesSkipped << dfmsEOL;
   	toLog << "---------------------------------------------------------------------- " << dfmsEOL;
   	toLog << "                          Total CPU Time: " << totTime << " s" << dfmsEOL;
   	toLog << "                     Actual Elapsed Time: " << rTotTime << " s" << dfmsEOL;
   	toLog << "      L2 Conversion Process Completed at: " << getProcessTime(3) << dfmsEOL;
   	toLog << "---------------------------------------------------------------------- " << dfmsEOL;
   	toLog.close();

   	// Optionally write to L3Info log file
   	if (L3INFOTOFILE) {
   		l3Log << dfmsEOL << dfmsEOL;
   		l3Log << "----------------------------------------------------------------------" << dfmsEOL;    	l3Log << "        L2 Conversion Process Started at: " << sTstart << dfmsEOL;
   		l3Log << "                Attempted L2 Conversions: " << totFilesAttempted << dfmsEOL;
   		l3Log << "               Successful L2 conversions: " << totFilesSuccessfullyProcessed << dfmsEOL;
   		l3Log << "---------------------------------------------------------------------- " << dfmsEOL;
   		l3Log << "          High Resolution Mode Exception: " << exceptionType["highRes"] << dfmsEOL;
   		l3Log << "    Peak Height Limit (Row  A) Exception: " << sideAexceptionType["peakHeightLimit"] << dfmsEOL;
   		l3Log << "    Peak Height Limit (Row  B) Exception: " << sideBexceptionType["peakHeightLimit"] << dfmsEOL;
   		l3Log << "   Peak Out of Bounds (Row  A) Exception: " << sideAexceptionType["peakOutOfBounds"] << dfmsEOL;
   		l3Log << "   Peak Out of Bounds (Row  B) Exception: " << sideBexceptionType["peakOutOfBounds"] << dfmsEOL;
   		l3Log << " No Useful peak found (Row  A) Exception: " << sideAexceptionType["peakFinderError"] << dfmsEOL;
   		l3Log << " No Useful peak found (Row  B) Exception: " << sideBexceptionType["peakFinderError"] << dfmsEOL;
   		l3Log << "---------------------------------------------------------------------- " << dfmsEOL;
   		l3Log << "   No GCU MPST file found - File Skipped: " << skipReas["noMPSTfile"] << dfmsEOL;
   		l3Log << "     GCU Mass Not in MPST - File Skipped: " << skipReas["gcuMassNotInMPST"] << dfmsEOL;
   		l3Log << "   Not an MCP or CEM file - File Skipped: " << skipReas["notMCPorCEM"] << dfmsEOL;
   		l3Log << "     High Resolution Mode - File Skipped: " << skipReas["highRes"] << dfmsEOL;
   		l3Log << "   ROSINA HK Mass is Zero - File Skipped: " << skipReas["rosMassZero"] << dfmsEOL;
   		l3Log << "        Peak Height Limit - File Skipped: " << skipReas["peakHeightLimit"] << dfmsEOL;
   		l3Log << "         Gain Not Defined - File Skipped: " << skipReas["gainNotDef"] << dfmsEOL;
   		l3Log << "       Pix0 out of Bounds - File Skipped: " << skipReas["pix0Bounds"] << dfmsEOL;
   		l3Log << "        Peak Finder Error - File Skipped: " << skipReas["peakFinderError"] << dfmsEOL;
   		l3Log << "  L2 HK with Line OverRun - File Skipped: " << skipReas["L2HKLineOverun"] << dfmsEOL;
   		l3Log << "L2 file in Exclusion Zone - File Skipped: " << skipReas["fileExcluded"] << dfmsEOL;
   		l3Log << "                  Total L2 Files Skipped: " << totFilesSkipped << dfmsEOL;
   		l3Log << "---------------------------------------------------------------------- " << dfmsEOL;
   		l3Log << "                          Total CPU Time: " << totTime << " s" << dfmsEOL;
   		l3Log << "                     Actual Elapsed Time: " << rTotTime << " s" << dfmsEOL;
   		l3Log << "      L2 Conversion Process Completed at: " << getProcessTime(3) << dfmsEOL;
   		l3Log << "---------------------------------------------------------------------- " << dfmsEOL;
    }
   	// Close L3Info log files (if requested)
   	if (l3Log.is_open()) l3Log.close();

   	// Write to STDOUT
   	cout << "\n\n----------------------------------------------------------------------" << endl;
   	cout << "        L2 Conversion Process Started at: " << sTstart << endl;
   	cout << "                Attempted L2 Conversions: " << totFilesAttempted << endl;
   	cout << "               Successful L2 conversions: " << totFilesSuccessfullyProcessed << endl;
   	cout << "---------------------------------------------------------------------- " << endl;
   	cout << "          High Resolution Mode Exception: " << exceptionType["highRes"] << endl;
   	cout << "    Peak Height Limit (Row  A) Exception: " << sideAexceptionType["peakHeightLimit"] << endl;
   	cout << "    Peak Height Limit (Row  B) Exception: " << sideBexceptionType["peakHeightLimit"] << endl;
   	cout << "   Peak Out of Bounds (Row  A) Exception: " << sideAexceptionType["peakOutOfBounds"] << endl;
   	cout << "   Peak Out of Bounds (Row  B) Exception: " << sideBexceptionType["peakOutOfBounds"] << endl;
   	cout << " No Useful peak found (Row  A) Exception: " << sideAexceptionType["peakFinderError"] << endl;
   	cout << " No Useful peak found (Row  B) Exception: " << sideBexceptionType["peakFinderError"] << endl;
   	cout << "---------------------------------------------------------------------- " << endl;
   	cout << "   No GCU MPST file found - File Skipped: " << skipReas["noMPSTfile"] << endl;
   	cout << "     GCU Mass Not in MPST - File Skipped: " << skipReas["gcuMassNotInMPST"] << endl;
   	cout << "   Not an MCP or CEM file - File Skipped: " << skipReas["notMCPorCEM"] << endl;
   	cout << "     High Resolution Mode - File Skipped: " << skipReas["highRes"] << endl;
   	cout << "   ROSINA HK Mass is Zero - File Skipped: " << skipReas["rosMassZero"] << endl;
   	cout << "        Peak Height Limit - File Skipped: " << skipReas["peakHeightLimit"] << endl;
   	cout << "         Gain Not Defined - File Skipped: " << skipReas["gainNotDef"] << endl;
   	cout << "       Pix0 out of Bounds - File Skipped: " << skipReas["pix0Bounds"] << endl;
   	cout << "        Peak Finder Error - File Skipped: " << skipReas["peakFinderError"] << endl;
   	cout << "  L2 HK with Line OverRun - File Skipped: " << skipReas["L2HKLineOverun"] << endl;
   	cout << "L2 file in Exclusion Zone - File Skipped: " << skipReas["fileExcluded"] << endl;
   	cout << "                  Total L2 Files Skipped: " << totFilesSkipped << endl;
   	cout << "---------------------------------------------------------------------- " << endl;
   	cout << "                          Total CPU Time: " << totTime << " s" << endl;
   	cout << "                     Actual Elapsed Time: " << rTotTime << " s" << endl;
   	cout << "      L2 Conversion Process Completed at: " << getProcessTime(3) << endl;
   	cout << "---------------------------------------------------------------------- " << endl;

    cout << "Number of CS2 calibrations: " << iicnt << endl;
    
   	if (QLINFOFILE) {
   		QuickLookLog << "--------------------------------------------------------------------------------------------------------------------------------------" << dfmsEOL;
    }

   	// Close QuikLook log files (if opened)
   	if (QuickLookLog.is_open()) QuickLookLog.close();

    // Close debug stream if open
    if (toDebug.is_open()) toDebug.close();
    
    // Dump some stats and then close file
    if (pix0D.is_open()) {
        pix0D << endl;
        pix0D << "---------------------------------------------" << endl;
        pix0D << " Pix0 RowA/B Delta Range    Pix0 Delta" << endl;
        pix0D << "---------------------------------------------" << endl;
        pix0D << "         Delta <=  3   =  " << pix0Spread["<=3"] << endl;
        pix0D << "     3 < Delta <=  6   =  " << pix0Spread["3 < pix0 <= 6"] << endl;
        pix0D << "     6 < Delta <= 10   =  " << pix0Spread["6 < pix0 <= 10"] << endl;
        pix0D << "         Delta  > 10   =  " << pix0Spread["pix0 > 10"] << endl;
        pix0D << "---------------------------------------------" << endl;
        pix0D.close();
    }
    
    // STEP 17 - Release dynamically allocated memory of local variables
	if (!DOCEM && pgObj != NULL) delete pgObj;
    if (L2DirObj != NULL) delete L2DirObj;
    if (gtObject != NULL) delete gtObject;
    if (modeIDobject != NULL) delete modeIDobject;
    
    // Exit successfully
    exit(EXIT_SUCCESS);
}
